<?php

declare(strict_types=1);

namespace Drupal\mediaflow\EventSubscriber;

use Drupal\csp\CspEvents;
use Drupal\csp\Event\PolicyAlterEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Csp Policy alter event subscriber.
 */
final class CspSubscriber implements EventSubscriberInterface {

  /**
   * Mediaflow domains.
   */
  private const DOMAINS = [
    'https://mfstatic.com',
    'https://api.mediaflow.com',
    'https://files.mediaflow.com',
    'https://play.mediaflow.com',
    'https://play.mediaflowpro.com',
    'https://*.mediaflow.com',
    'https://*.inviewer.se',
  ];

  /**
   * {@inheritDoc}
   */
  public static function getSubscribedEvents(): array {
    $events = [];
    if (class_exists('Drupal\csp\CspEvents')) {
      $events = [
        CspEvents::POLICY_ALTER => 'alterPolicy',
      ];
    }
    return $events;
  }

  /**
   * Fires an event on csp policy alter event.
   *
   * @param \Drupal\csp\Event\PolicyAlterEvent $event
   *   Event object.
   */
  public function alterPolicy(PolicyAlterEvent $event): void {
    $policy = $event->getPolicy();
    $directives = [
      'connect-src',
      'img-src',
      'frame-src',
      'font-src',
    ];
    foreach ($directives as $directive) {
      if ($policy->hasDirective($directive)) {
        $policy->appendDirective($directive, self::DOMAINS);
      }
      elseif ($policy->hasDirective('default-src')) {
        $connectDirective = array_merge($policy->getDirective('default-src'), self::DOMAINS);
        $policy->setDirective($directive, $connectDirective);
      }
    }
  }

}
