<?php

namespace Drupal\mediaflow\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\media\Entity\MediaType;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Implements Mediaflow settings form.
 */
class MediaflowSettingsForm extends ConfigFormBase {

  const METHODS = [
    'js' => 'Javascript',
    'iframe' => 'Iframe',
    'choose' => 'Choose in media selector',
  ];

  /**
   * The Mediaflow fetcher.
   *
   * @var \Drupal\mediaflow\Service\MediaflowFetcherInterface
   */
  protected $fetcher;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->fetcher = $container->get('mediaflow.fetcher');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['mediaflow.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'mediaflow_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('mediaflow.settings');

    $form = parent::buildForm($form, $form_state);

    $form['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID'),
      '#default_value' => $config->get('client_id'),
    ];
    $form['client_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client secret'),
      '#default_value' => $config->get('client_secret'),
    ];
    $form['refresh_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Refresh token'),
      '#default_value' => $config->get('refresh_token'),
    ];

    $form['set_alt_text'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enforce alt-texts'),
      '#default_value' => $config->get('set_alt_text'),
    ];

    $form['method'] = [
      '#type' => 'select',
      '#title' => $this->t('Video embed method.'),
      '#default_value' => $config->get('method'),
      '#options' => $this->methodOptions(),
    ];

    $status = $this->fetcher->getAccessToken();
    if ($status) {
      $list_items = [];
      $link = Link::createFromRoute($this->t('Text formats'), 'filter.admin_overview');
      $list_items[] = $this->t('Adding Mediaflow button to your ckeditor toolbar for each text format: @link', [
        '@link' => $link->toString(),
      ]);
      $link = Link::createFromRoute($this->t('Content types'), 'entity.node_type.collection');
      $list_items[] = $this->t('Adding Media fields to your Content types: @link', [
        '@link' => $link->toString(),
      ]);
      /** @var \Drupal\media\MediaTypeInterface $type */
      $type = MediaType::load('mediaflow');
      if ($type) {
        $link = $type->toLink($this->t('Mediaflow media type'), 'edit-form');
        $list_items[] = $this->t('Configuring your Mediaflow Media Type: @link', [
          '@link' => $link->toString(),
        ]);
      }

      $form['info'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Status'),
        '#weight' => 98,
        [
          [
            '#type' => 'html_tag',
            '#tag' => 'p',
            '#value' => $this->t('Your connection to Mediaflow works.'),
          ],
          [
            '#theme' => 'item_list',
            '#items' => $list_items,
            '#title' => $this->t('Your next steps may include:'),
            '#list_type' => 'ul',
            '#attributes' => [],
          ],
        ],
      ];
    }
    elseif (!empty($config->get('refresh_token'))) {
      $form['info'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Status'),
        '#weight' => 98,
        [
          [
            '#type' => 'html_tag',
            '#tag' => 'p',
            '#value' => $this->t('Connection could not be established. Check your access token.'),
          ],
        ],
      ];
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('mediaflow.settings')
      ->set('client_id', $form_state->getValue('client_id'))
      ->set('client_secret', $form_state->getValue('client_secret'))
      ->set('refresh_token', $form_state->getValue('refresh_token'))
      ->set('allow_crop', $form_state->getValue('allow_crop'))
      ->set('set_alt_text', $form_state->getValue('set_alt_text'))
      ->set('method', $form_state->getValue('method'))
      ->save();

    // Clear stored access token.
    $this->fetcher->clearAccessToken();

    parent::submitForm($form, $form_state);
  }

  /**
   * Returns video method options.
   *
   * @return array
   *   Method options.
   */
  private function methodOptions() {
    return array_map('t', self::METHODS);
  }

}
