<?php

namespace Drupal\mediaflow\Plugin\CKEditor5Plugin;

use Drupal\ckeditor5\Plugin\CKEditor5PluginConfigurableInterface;
use Drupal\ckeditor5\Plugin\CKEditor5PluginConfigurableTrait;
use Drupal\ckeditor5\Plugin\CKEditor5PluginDefault;
use Drupal\ckeditor5\Plugin\CKEditor5PluginDefinition;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\editor\EditorInterface;
use Drupal\media\Plugin\media\Source\Image;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * CKEditor5 Mediaflow plugin configuration.
 */
class Mediaflow extends CKEditor5PluginDefault implements CKEditor5PluginConfigurableInterface, ContainerFactoryPluginInterface {

  use CKEditor5PluginConfigurableTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The "mediaflow.settings" configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $config;

  /**
   * Mediaflow constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param \Drupal\ckeditor5\Plugin\CKEditor5PluginDefinition $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    CKEditor5PluginDefinition $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    ConfigFactoryInterface $config_factory,
    AccountInterface $current_user,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->config = $config_factory->get('mediaflow.settings');
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDynamicPluginConfig(array $static_plugin_config, EditorInterface $editor): array {
    $static_plugin_config['mediaflow'] = [
      'tokenEndpoint' => Url::fromRoute('mediaflow.token')->toString(),
      'locale' => $this->currentUser->getPreferredAdminLangcode() != 'sv' ? 'en_US' : 'sv_SE',
      'downloadFormat' => 'mediaflow',
      'limitFileType' => 'image',
      'forceAltText' => $this->config->get('set_alt_text'),
    ] + $this->configuration;
    return $static_plugin_config;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return ['media_bundle' => ''];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $bundles = $this->entityTypeManager
      ->getStorage('media_type')
      ->loadMultiple();
    $options = array_map(function ($bundle) {
      if (!($bundle->getSource() instanceof Image)) {
        return NULL;
      }
      return sprintf('%s (%s)', $bundle->label(), $bundle->id());
    }, $bundles);
    $options = ['' => $this->t('- Select -')] + array_filter($options);
    $form['media_bundle'] = [
      '#type' => 'select',
      '#title' => $this->t('Media type'),
      '#options' => $options,
      '#default_value' => $this->configuration['media_bundle'],
      '#label_attributes' => [
        'class' => ['form-required'],
      ],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $media_bundle = $form_state->getValue('media_bundle');
    if (empty($media_bundle)) {
      $form_state->setErrorByName('media_bundle', $this->t('Please choose Media type under the Mediaflow settings.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['media_bundle'] = $form_state->getValue('media_bundle');
  }

}
