<?php

namespace Drupal\mediaflow\Plugin\Field\FieldWidget;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\StringTextfieldWidget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\file\FileInterface;
use Drupal\media\Entity\MediaType;
use Drupal\mediaflow\Plugin\media\Source\Mediaflow;
use Drupal\mediaflow\Service\UsageManager;
use Drupal\mediaflow\Service\UsageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'mediaflow_integer' widget.
 *
 * @FieldWidget(
 *   id = "mediaflow_integer",
 *   label = @Translation("Mediaflow Selector"),
 *   field_types = {
 *     "integer",
 *   },
 * )
 */
class MediaflowWidget extends StringTextfieldWidget {

  const METHODS = [
    'default' => 'Use site setting',
    'choose' => 'Choose for each resource',
    'iframe' => 'Iframe',
    'js' => 'Javascript',
  ];

  const MEDIA_TYPES = [
    'image,video' => 'Both images and video',
    'image' => 'Images',
    'video' => 'Video',
  ];

  /**
   * The mediaflow settings config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The usage manager.
   *
   * @var \Drupal\mediaflow\Service\UsageManager
   */
  protected $usageManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    EntityTypeManagerInterface $entity_type_manager,
    ConfigFactoryInterface $config_factory,
    UsageManager $usage_manager,
    AccountProxyInterface $current_user,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->config = $config_factory->get('mediaflow.settings');
    $this->usageManager = $usage_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
      $container->get('mediaflow.usage_manager'),
      $container->get('current_user'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'method' => 'default',
      'media_type' => 'image,video',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form['media_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Allowed media types'),
      '#default_value' => $this->getSetting('media_type'),
      '#options' => $this->mediaTypeOptions(),
    ];
    $form['method'] = [
      '#type' => 'select',
      '#title' => $this->t('Embed method for video.'),
      '#default_value' => $this->getSetting('method'),
      '#options' => $this->methodOptions(),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary[] = t('Allowed media: @media_type', [
      '@media_type' => $this->mediaTypeOptions()[$this->getSetting('media_type')],
    ]);
    if ($this->getSetting('media_type') !== 'image') {
      $summary[] = t('Embed method: @method', [
        '@method' => $this->methodOptions()[$this->getSetting('method')],
      ]);
    }
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element['#access'] = $this->currentUser->hasPermission('use mediaflow');
    $element['value'] = $element;
    $value = $items[$delta]->value ?? '';
    if (!empty($value)) {
      $element['preview'] = $this->viewElement($items, $delta, 'en');
      $element['value']['#type'] = 'value';
      $element['value']['#value'] = $value;
    }
    else {
      $method = $this->getSetting('method');
      if ($method === 'default') {
        $method = $this->config->get('method');
      }
      $element['value']['#prefix'] = '<div class="mediaflow-widget-wrapper">';
      $element['value']['#suffix'] = '</div>';
      $element['value']['#type'] = 'mediaflow_selector';
      $element['value']['#media_type'] = $this->getSetting('media_type');
      $element['value']['#method'] = $method;
    }
    return $element;
  }

  /**
   * View element.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   Field items.
   * @param int $delta
   *   The item delta.
   * @param string $langcode
   *   Language code.
   *
   * @return array
   *   Render array.
   */
  public function viewElement(FieldItemListInterface $items, int $delta, string $langcode) {
    $value = $items->offsetGet($delta)->getValue();
    $value = $value['value'];
    $type = $this->usageManager->getValue($value, 'type');
    if (((int) $type) === UsageManagerInterface::EMBED) {
      $html = $this->usageManager->getValue($value, 'html');
      return [
        '#theme' => 'mediaflow_asset_formatter',
        '#type' => 'video',
        '#tag' => 'div',
        '#code_embed' => $html,
        '#attributes' => [
          'data-mediaflow-id' => $value,
          'class' => [
            'mediaflow-video-placeholder',
          ],
        ],
      ];
    }
    if (((int) $type) === UsageManagerInterface::FILE) {
      $fid = $this->usageManager->getValue($value, 'file');
      $data = $this->usageManager->getValue($value, 'data');
      $data = unserialize($data);
      $storage = $this->entityTypeManager->getStorage('file');
      $file = $storage->load($fid);
      if ($file instanceof FileInterface) {
        return [
          '#theme' => 'mediaflow_asset_formatter',
          '#type' => 'image',
          '#uri' => $file->createFileUrl(),
          '#alt' => $data['altText'],
          '#width' => $data['width'],
          '#height' => $data['height'],
          '#style_name' => 'large',
        ];
      }
    }
    return [
      '#markup' => t('Malformed Mediaflow Entity'),
    ];
  }

  /**
   * View image.
   */
  public function viewImage(FieldItemInterface $item, FileInterface $file) {
    return [
      '#theme' => 'mediaflow_asset_formatter',
      '#item' => $item,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    $target_bundle = $field_definition->getTargetBundle();
    if (!parent::isApplicable($field_definition) || $field_definition->getTargetEntityTypeId() !== 'media' || !$target_bundle) {
      return FALSE;
    }
    return MediaType::load($target_bundle)->getSource() instanceof Mediaflow;
  }

  /**
   * Get methods.
   *
   * @return array
   *   Methods list.
   */
  private function methodOptions() {
    return array_map('t', self::METHODS);
  }

  /**
   * Get media types.
   *
   * @return array
   *   Media types list.
   */
  private function mediaTypeOptions() {
    return array_map('t', self::MEDIA_TYPES);
  }

}
