<?php

namespace Drupal\mediaflow\Plugin\media\Source;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\Display\EntityFormDisplayInterface;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldTypePluginManagerInterface;
use Drupal\media\MediaInterface;
use Drupal\media\MediaSourceBase;
use Drupal\media\MediaSourceInterface;
use Drupal\media\MediaTypeInterface;
use Drupal\mediaflow\Service\UsageManager;
use Drupal\mediaflow\Service\UsageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a media source plugin for Mediaflow Image/Video.
 *
 * @MediaSource(
 *   id = "mediaflow",
 *   label = @Translation("Mediaflow"),
 *   description = @Translation("Embedded Mediaflow Image or Video."),
 *   allowed_field_types = {"mediaflow_item"},
 *   default_thumbnail_filename = "no-thumbnail.png",
 *   providers = {},
 * )
 */
class Mediaflow extends MediaSourceBase implements MediaSourceInterface {

  /**
   * The usage manager.
   *
   * @var \Drupal\mediaflow\Service\UsageManager
   */
  protected $usageManager;

  /**
   * Constructs a new Mediaflow instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Field\FieldTypePluginManagerInterface $field_type_manager
   *   The field type plugin manager service.
   * @param \Drupal\mediaflow\Service\UsageManager $usage_manager
   *   The usage manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager, ConfigFactoryInterface $config_factory, FieldTypePluginManagerInterface $field_type_manager, UsageManager $usage_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entity_type_manager, $entity_field_manager, $field_type_manager, $config_factory);
    $this->usageManager = $usage_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager'),
      $container->get('config.factory'),
      $container->get('plugin.manager.field.field_type'),
      $container->get('mediaflow.usage_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getMetadataAttributes() {
    return [
      'name' => $this->t('Mediaflow resource name'),
      'html' => $this->t('Embedded video code'),
      'filetype' => $this->t('File type'),
      'altText' => $this->t('Alt-text'),
      'width' => $this->t('Width'),
      'height' => $this->t('Height'),
      'photographer' => $this->t('Photographer'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getMetadata(MediaInterface $media, $attribute_name) {
    $source = $media->getSource();
    if ($source instanceof Mediaflow) {
      $id = $source->getSourceFieldValue($media);
      if ($attribute_name == 'thumbnail_uri') {
        return $this->getThumbnailUri($id) ?: parent::getMetadata($media, $attribute_name);
      }
      return $this->usageManager->getValue($id, $attribute_name);
    }
    return NULL;
  }

  /**
   * Get Mediaflow thumbnail uri.
   *
   * @param string $id
   *   Mediaflow ID.
   *
   * @return false|string|null
   *   File uri.
   */
  public function getThumbnailUri(string $id) {
    $type = $this->usageManager->getValue($id, 'type');
    if ($type == UsageManagerInterface::FILE) {
      $fid = $this->usageManager->getValue($id, 'file');
      if ($fid) {
        $storage = $this->entityTypeManager->getStorage('file');
        /** @var \Drupal\file\FileInterface $file */
        $file = $storage->load($fid);
        return $file->getFileUri();
      }
    }

    // Fallback to getting file type icon from media module.
    $basetype = $this->usageManager->getValue($id, 'basetype');
    $filetype = $this->usageManager->getValue($id, 'filetype');
    $icon_base = $this->configFactory->get('media.settings')->get('icon_base_uri');

    $icon_names = [
      $basetype . '--' . $filetype,
      $filetype,
      $basetype,
    ];

    // Get icon if any is found for filetype.
    foreach ($icon_names as $icon_name) {
      $thumbnail = $icon_base . '/' . $icon_name . '.png';
      if (is_file($thumbnail)) {
        return $thumbnail;
      }
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareViewDisplay(MediaTypeInterface $type, EntityViewDisplayInterface $display) {
    $display->setComponent($this->getSourceFieldDefinition($type)->getName(), [
      'type' => 'mediaflow',
      'label' => 'visually_hidden',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function prepareFormDisplay(MediaTypeInterface $type, EntityFormDisplayInterface $display) {
    parent::prepareFormDisplay($type, $display);
    $source_field = $this->getSourceFieldDefinition($type)->getName();
    $display->setComponent($source_field, [
      'type' => 'mediaflow_integer',
      'weight' => $display->getComponent($source_field)['weight'],
    ]);
    $display->removeComponent('name');
  }

  /**
   * {@inheritdoc}
   */
  public function createSourceField(MediaTypeInterface $type) {
    $plugin_definition = $this->getPluginDefinition();
    $label = (string) $this->t('@type media', [
      '@type' => $plugin_definition['label'],
    ]);
    return parent::createSourceField($type)->set('label', $label);
  }

}
