<?php

namespace Drupal\mediaflow\Service;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\KeyValueStore\KeyValueExpirableFactoryInterface;
use Drupal\file\FileRepositoryInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\RequestOptions;

/**
 * Implements Mediaflow Fetcher service.
 */
class MediaflowFetcher implements MediaflowFetcherInterface {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $client;

  /**
   * The file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  protected $fileRepository;

  /**
   * The "mediaflow.settings" configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $config;

  /**
   * Access Token store.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueStoreInterface|mixed
   */
  private $accessTokenStore;

  /**
   * Constructor.
   *
   * @param \GuzzleHttp\Client $client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactory $config_factory
   *   The config factory.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system.
   * @param \Drupal\file\FileRepositoryInterface $file_repository
   *   The file repository service.
   * @param \Drupal\Core\KeyValueStore\KeyValueExpirableFactoryInterface $key_value_expirable
   *   The key value expire.
   */
  public function __construct(
    Client $client,
    ConfigFactory $config_factory,
    FileSystemInterface $file_system,
    FileRepositoryInterface $file_repository,
    KeyValueExpirableFactoryInterface $key_value_expirable,
  ) {
    $this->client = $client;
    $this->config = $config_factory->get('mediaflow.settings');
    $this->fileSystem = $file_system;
    $this->fileRepository = $file_repository;
    $this->accessTokenStore = $key_value_expirable->get('mediaflow.access_key_store');
  }

  /**
   * {@inheritdoc}
   */
  public function downloadFile(array $data) {
    $client = new Client();
    $response = $client->get($data['url'], ['verify' => FALSE]);

    if ($response->getStatusCode() == 200) {
      $ext = $data['filetype'];
      $filename = $data['name'] . '_' . date('Y-m-d--H-i-s') . '.' . $ext;
      if ($filename) {
        $file_data = $response->getBody()->getContents();
        $directory = 'public://mediaflow/';
        $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY);
        $file = $this->fileRepository->writeData($file_data, $directory . $filename, FileExists::Replace);
        return $file;
      }
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function updateUsages(array $data) {
    try {
      $response = $this->post('/files/multiple/usages', [
        'access_token' => $this->getAccessToken(),
      ], $data);
      return $response;
    }
    catch (\Throwable $e) {
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deleteUsage(array $data) {
    try {
      $response = $this->post('/files/multiple/usages', [
        'access_token' => $this->getAccessToken(),
      ], $data);
      return $response;
    }
    catch (\Throwable $e) {
      return [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getAccessToken(bool $refresh = FALSE) {
    if ($refresh) {
      $this->renewAccessToken();
    }
    $token = $this->accessTokenStore->get('access_token');
    $expire = $token['expires_in'] ?? 0;
    // Renew token if it valid less than 5 min.
    if (time() > ($expire - 300)) {
      $token['token'] = NULL;
      if ($this->renewAccessToken()) {
        $token = $this->accessTokenStore->get('access_token');
      }
    }
    return $token['token'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function clearAccessToken() {
    $this->accessTokenStore->delete('access_token');
  }

  /**
   * {@inheritdoc}
   */
  public function ensureAccess() {
    $this->renewAccessToken();
  }

  /**
   * Renews access tokens.
   *
   * @return bool
   *   TRUE on success, otherwise FALSE.
   */
  protected function renewAccessToken() {
    try {
      $response = $this->get('/oauth2/token', [
        'grant_type' => 'refresh_token',
        'client_id' => $this->config->get('client_id'),
        'client_secret' => $this->config->get('client_secret'),
        'refresh_token' => $this->config->get('refresh_token'),
      ]);
      if (!empty($response['access_token'])) {
        $token = [
          'token' => $response['access_token'],
          'expires_in' => time() + $response['expires_in'],
        ];
        $this->accessTokenStore->setWithExpire('access_token', $token, $response['expires_in']);
        return TRUE;
      }
    }
    catch (ClientException $exception) {
    }
    return FALSE;
  }

  /**
   * Performs a GET request to the api.
   *
   * @param string $resource
   *   API resource.
   * @param array $query
   *   Request query.
   * @param array $body
   *   Request body.
   *
   * @return array
   *   Response.
   */
  private function get(string $resource, array $query = [], array $body = []) {
    $headers = [];

    if (!empty($query['access_token'])) {
      $headers['Authorization'] = 'Bearer ' . $query['access_token'];
      unset($query['access_token']);
    }
    $options = [
      RequestOptions::HEADERS => $headers,
    ];
    if (!empty($query)) {
      $options[RequestOptions::QUERY] = $query;
    }
    $response = $this->client->get(self::API_URL . $resource, $options);
    return Json::decode($response->getBody());
  }

  /**
   * Performs a POST request to the api.
   *
   * @param string $resource
   *   API resource.
   * @param array $query
   *   Request query.
   * @param array $body
   *   Request body.
   *
   * @return array
   *   Response.
   */
  private function post(string $resource, array $query = [], array $body = []) {
    $headers = [];
    if (!empty($query['access_token'])) {
      $headers['Authorization'] = 'Bearer ' . $query['access_token'];
      unset($query['access_token']);
    }

    // Fix bool values.
    if (isset($body['removed'])) {
      $body['removed'] = $body['removed'] ? 'true' : 'false';
    }

    // Make sure ids are integers.
    if (isset($body['id']) && is_array($body['id'])) {
      foreach ($body['id'] as $key => $value) {
        $body['id'][$key] = (int) $value;
      }
    }

    $headers['Content-Type'] = 'application/json';
    $options = [
      RequestOptions::HEADERS => $headers,
      RequestOptions::BODY => json_encode($body),
    ];
    if (!empty($query)) {
      $options[RequestOptions::QUERY] = $query;
    }
    $response = $this->client->post(self::API_URL . $resource, $options);

    return Json::decode($response->getBody());
  }

}
