<?php

namespace Drupal\mediaflow\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManager;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\media\MediaInterface;
use Drupal\node\NodeInterface;

/**
 * Implements Mediaflow Usage manager.
 */
class UsageManager implements UsageManagerInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private $entityTypeManager;

  /**
   * The mediaflow fetcher.
   *
   * @var \Drupal\mediaflow\Service\MediaflowFetcher
   */
  private $fetcher;

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  private $database;

  /**
   * The field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManager
   */
  private $entityFieldManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  private $moduleHandler;

  /**
   * The site config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  private $siteConfig;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Entity\EntityFieldManager $entity_field_manager
   *   The field manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   Database connection.
   * @param \Drupal\mediaflow\Service\MediaflowFetcher $fetcher
   *   The mediaflow fetcher.
   * @param \Drupal\Core\Extension\ModuleHandler $module_handler
   *   The module handler.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    EntityFieldManager $entity_field_manager,
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database,
    MediaflowFetcher $fetcher,
    ModuleHandler $module_handler,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->entityFieldManager = $entity_field_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->fetcher = $fetcher;
    $this->database = $database;
    $this->moduleHandler = $module_handler;
    $this->siteConfig = $config_factory->get('system.site');
  }

  /**
   * {@inheritdoc}
   */
  public function getFetcher() {
    return $this->fetcher;
  }

  /**
   * {@inheritdoc}
   */
  public function downloadUri(&$data) {
    if (empty($data['mediaType']) || $data['mediaType'] !== 'video') {
      $data['file'] = $this->fetcher->downloadFile($data);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function addMedia(array $data) {
    $values = [];
    if (isset($data['embedCode'])) {
      $values = [
        'mediaflow_origin_id' => $data['id'],
        'data' => serialize($data),
        'html' => $data['embedCode'],
        'type' => self::EMBED,
      ];
    }
    elseif (!empty($data['url']) && empty($data['file'])) {
      $this->downloadUri($data);
      if (isset($data['file'])) {
        $values = [
          'mediaflow_origin_id' => $data['id'],
          'file' => $data['file']->id(),
          'type' => self::FILE,
        ];
        // Do not serialize file entity.
        unset($data['file']);
        $values['data'] = serialize($data);
      }
    }

    return $this->saveMediaData($values);
  }

  /**
   * {@inheritdoc}
   */
  public function saveMediaData(array $data) {
    if (!empty($data)) {
      try {
        return $this->database->insert('mediaflow')
          ->fields($data)
          ->execute();
      }
      catch (\Throwable $e) {
        return FALSE;
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(string $id, string $key) {
    if ($this->database->schema()->fieldExists('mediaflow', $key)) {
      return $this->database->select('mediaflow', 'fm')
        ->fields('fm', [$key])
        ->condition('fm.id', $id)
        ->execute()
        ->fetchField();
    }
    else {
      $data = unserialize($this->getValue($id, 'data'));
      if ($key === 'name' && $data['basetype'] === 'video') {
        $key = 'filename';
      }
      if (isset($data[$key])) {
        return $data[$key];
      }
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function onEntityCreate(EntityInterface $entity) {
    /** @var \Drupal\node\NodeInterface[] $nodes */
    /** @var \Drupal\media\MediaInterface[] $medias */
    [$nodes, $medias] = $this->discoverEntities($entity);
    foreach ($nodes as $node) {
      foreach ($medias as $media) {
        if (is_array($media) && isset($media['value'])) {
          $ids[] = $media['value'];
        }
        else {
          $id = $this->getMediaflowIdFromMedia($media);
          if ($id) {
            $ids[] = $id;
          }
        }
      }

      if (empty($ids)) {
        continue;
      }

      $mf_ids = $this->database->select('mediaflow', 'fm')
        ->fields('fm', ['mediaflow_origin_id'])
        ->condition('fm.id', $ids, 'IN')
        ->execute()->fetchCol();
      $url = $node->toUrl('canonical', ['absolute' => TRUE])->toString();
      $nid = $node->id();

      if (count($mf_ids)) {
        $data = [
          'id' => $mf_ids,
          'contact' => $node->getOwner()->label(),
          'project' => $this->siteConfig->get('name') ?: $this->t('Drupal site'),
          'date' => date('Y-m-d H:i:s'),
          'amount' => '1',
          'description' => '',
          'types' => ['web'],
          'web' => [
            'page' => $url,
            'pageName' => $node->label(),
          ],
          'removed' => FALSE,
        ];

        $jsonResponse = $this->fetcher->updateUsages($data);
        if (isset($jsonResponse['status']) && $jsonResponse['status'] == 201) {
          $this->database->update('mediaflow')
            ->fields(['usage_reported' => 1, 'entity_id' => $nid])
            ->condition('id', $ids, 'IN')
            ->condition('mediaflow_origin_id', $mf_ids, 'IN')
            ->execute();
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onEntityUpdate(EntityInterface $entity) {
    /** @var \Drupal\node\NodeInterface[] $nodes */
    /** @var \Drupal\media\MediaInterface[] $medias */
    [$nodes, $medias] = $this->discoverEntities($entity);
    foreach ($nodes as $node) {
      foreach ($medias as $media) {
        if (is_array($media) && isset($media['value'])) {
          $ids[] = $media['value'];
        }
        else {
          $id = $this->getMediaflowIdFromMedia($media);
          if ($id) {
            $ids[] = $id;
          }
        }
      }

      if (empty($ids)) {
        continue;
      }

      $mf_ids = $this->database->select('mediaflow', 'fm')
        ->fields('fm', ['mediaflow_origin_id'])
        ->condition('fm.id', $ids, 'IN')
        ->condition('fm.usage_reported', 0)
        ->execute()->fetchCol();
      $url = $node->toUrl('canonical', ['absolute' => TRUE])->toString();
      $nid = $node->id();

      if (count($mf_ids)) {
        $data = [
          'id' => $mf_ids,
          'contact' => $node->getOwner()->label(),
          'project' => $this->siteConfig->get('name') ?: $this->t('Drupal site'),
          'date' => date('Y-m-d H:i:s'),
          'amount' => '1',
          'description' => '',
          'types' => ['web'],
          'web' => [
            'page' => $url,
            'pageName' => $node->label(),
          ],
          'removed' => FALSE,
        ];

        $jsonResponse = $this->fetcher->updateUsages($data);
        if (isset($jsonResponse['status']) && $jsonResponse['status'] == 201) {
          $this->database->update('mediaflow')
            ->fields(['usage_reported' => 1, 'entity_id' => $nid])
            ->condition('id', $ids, 'IN')
            ->condition('mediaflow_origin_id', $mf_ids, 'IN')
            ->execute();
        }
      }

      $removed_mf_ids = $this->database->select('mediaflow', 'fm')
        ->fields('fm', ['mediaflow_origin_id'])
        ->condition('fm.id', $ids, 'NOT IN')
        ->condition('fm.usage_reported', 1)
        ->condition('fm.entity_id', $nid)
        ->execute()->fetchCol();

      if (count($removed_mf_ids)) {
        $data = [
          'id' => $removed_mf_ids,
          'types' => ['web'],
          'amount' => '1',
          'web' => [
            'page' => $url,
            'pageName' => $node->label(),
          ],
          'project' => $url,
          'removed' => TRUE,
        ];

        $jsonResponse = $this->fetcher->deleteUsage($data);
        if (isset($jsonResponse['status']) && $jsonResponse['status'] == 201) {
          $this->database->update('mediaflow')
            ->fields(['usage_reported' => 0, 'entity_id' => 0])
            ->condition('mediaflow_origin_id', $removed_mf_ids, 'IN')
            ->condition('entity_id', $nid)
            ->execute();
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onEntityDelete(EntityInterface $entity) {
    /** @var \Drupal\node\NodeInterface[] $nodes */
    /** @var \Drupal\media\MediaInterface[] $medias */
    [$nodes, $medias] = $this->discoverEntities($entity);
    foreach ($nodes as $node) {
      foreach ($medias as $media) {
        if (is_array($media) && isset($media['value'])) {
          $ids[] = $media['value'];
        }
        else {
          $id = $this->getMediaflowIdFromMedia($media);
          if ($id) {
            $ids[] = $id;
          }
        }
      }

      if (empty($ids)) {
        continue;
      }

      $mf_ids = $this->database->select('mediaflow', 'fm')
        ->fields('fm', ['mediaflow_origin_id'])
        ->condition('fm.id', $ids, 'IN')
        ->condition('fm.usage_reported', 1)
        ->execute()->fetchCol();
      $url = $node->toUrl('canonical', ['absolute' => TRUE])->toString();

      if (count($mf_ids)) {
        $data = [
          'id' => $mf_ids,
          'types' => ['web'],
          'amount' => '1',
          'web' => [
            'page' => $url,
            'pageName' => $node->label(),
          ],
          'project' => $url,
          'removed' => TRUE,
        ];

        $jsonResponse = $this->fetcher->deleteUsage($data);
        if (isset($jsonResponse['status']) && $jsonResponse['status'] == 201) {
          $this->database->update('mediaflow')
            ->fields(['usage_reported' => 0, 'entity_id' => 0])
            ->condition('id', $ids, 'IN')
            ->execute();
        }
      }
    }
  }

  /**
   * Discovers entities for Mediaflow media.
   *
   * Takes an entity being altered and discovers what files and what nodes
   * are affected by the change.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   Entity to discover.
   *
   * @return array
   *   List of nodes and media entities.
   */
  private function discoverEntities(EntityInterface $entity) {
    $medias = [];
    $nodes = [];
    if ($entity instanceof NodeInterface) {
      $medias = $this->mediaInNode($entity);
      $nodes = [$entity];
    }
    if ($entity instanceof MediaInterface) {
      $medias = [$entity];
      $nodes = $this->nodesContainingMedia($entity);
    }
    return [$nodes, $medias];
  }

  /**
   * Finds any files nested in this node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   Node entity.
   *
   * @return array|\Drupal\media\MediaInterface[]
   *   List with Mediaflow IDs or media entities.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function mediaInNode(NodeInterface $node) {
    $media = [];
    if ($this->moduleHandler->moduleExists('paragraphs')) {
      $paragraphs = $this->getReferencedEntities($node, 'paragraph');
      foreach ($paragraphs as $paragraph) {
        array_push($media, ...$this->getReferencedMediaflow($paragraph));
        array_push($media, ...$this->getEmbeddedMediaEntities($paragraph));
      }
    }
    array_push($media, ...$this->getReferencedMediaflow($node));
    array_push($media, ...$this->getEmbeddedMediaEntities($node));

    return $media;
  }

  /**
   * Finds any nodes containing given media.
   *
   * @param \Drupal\media\MediaInterface $media
   *   Media entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface[]
   *   List with node entities.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function nodesContainingMedia(MediaInterface $media) {
    $nodes = [];
    if ($this->moduleHandler->moduleExists('paragraphs')) {
      $paragraphs = $this->getEntitiesWithEmbededMedia($media, 'paragraph');
      array_push($paragraphs, ...$this->getReferencingEntities($media, 'paragraph'));
      foreach ($paragraphs as $paragraph) {
        array_push($nodes, ...$this->getReferencingEntities($paragraph, 'node'));
      }
    }
    array_push($nodes, ...$this->getEntitiesWithEmbededMedia($media, 'node'));
    array_push($nodes, ...$this->getReferencingEntities($media, 'node'));
    return $nodes;
  }

  /**
   * Finds any entities containing this entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to discover.
   * @param string $entity_type
   *   Entity type ID.
   *
   * @return \Drupal\Core\Entity\EntityInterface[]
   *   List of entities.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function getReferencingEntities(EntityInterface $entity, string $entity_type) {
    /** @var \Drupal\Core\Field\FieldStorageDefinitionInterface[] $fieldDefinitions */
    $fieldDefinitions = $this->entityFieldManager->getFieldStorageDefinitions($entity_type);
    $reference_fields = array_filter($fieldDefinitions, function ($definition) use ($entity) {
      return in_array($definition->getType(), self::REFERENCE_FIELD_TYPES)
        && $definition->getSetting('target_type') == $entity->getEntityTypeId();
    });
    if (!count($reference_fields)) {
      return [];
    }
    $storage = $this->entityTypeManager->getStorage($entity_type);
    $query = $storage->getQuery('OR');
    foreach ($reference_fields as $definition) {
      $query->condition($definition->getName(), $entity->id());
    }
    $ids = $query->accessCheck()->execute();
    if (!count($ids)) {
      return [];
    }
    return $storage->loadMultiple($ids);
  }

  /**
   * Finds any entities with this media embedded in it's wysiwyg fields.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to discover.
   * @param string $entity_type
   *   Entity type ID.
   *
   * @return \Drupal\Core\Entity\EntityInterface[]
   *   List of entities.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function getEntitiesWithEmbededMedia(EntityInterface $entity, string $entity_type) {
    /** @var \Drupal\Core\Field\FieldStorageDefinitionInterface[] $fieldDefinitions */
    $fieldDefinitions = $this->entityFieldManager->getFieldStorageDefinitions($entity_type);
    $reference_fields = array_filter($fieldDefinitions, function ($definition) {
      return in_array($definition->getType(), self::MEDIA_EMBED_FIELD_TYPES);
    });
    if (!count($reference_fields)) {
      return [];
    }
    $storage = $this->entityTypeManager->getStorage($entity_type);
    $query = $storage->getQuery('OR');
    foreach ($reference_fields as $definition) {
      $media_string = 'data-entity-uuid="' . $entity->uuid() . '"';
      $query->condition($definition->getName(), $media_string, 'CONTAINS');
    }
    $ids = $query->accessCheck()->execute();
    if (!count($ids)) {
      return [];
    }
    return $storage->loadMultiple($ids);
  }

  /**
   * Finds any entities nested in this entity.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity to discover.
   * @param string $entity_type
   *   Entity type ID.
   *
   * @return \Drupal\Core\Entity\FieldableEntityInterface[]
   *   List of entities.
   */
  private function getReferencedEntities(FieldableEntityInterface $entity, string $entity_type) {
    $reference_fields = array_filter($entity->getFieldDefinitions(), function ($definition) use ($entity_type) {
      return in_array($definition->getType(), self::REFERENCE_FIELD_TYPES) && $definition->getSetting('target_type') === $entity_type;
    });
    if (!count($reference_fields)) {
      return [];
    }

    $targets = [];
    foreach ($reference_fields as $definition) {
      $name = $definition->getName();
      if ($entity->hasField($name)) {
        /** @var \Drupal\Core\Field\EntityReferenceFieldItemList $field_item */
        $field_item = $entity->get($name);
        array_push($targets, ...$field_item->referencedEntities());
      }
    }
    return $targets;
  }

  /**
   * Finds any Mediaflow references in this entity.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity to discover.
   *
   * @return array
   *   List of Mediaflow IDs.
   */
  private function getReferencedMediaflow(FieldableEntityInterface $entity) {
    $reference_fields = array_filter($entity->getFieldDefinitions(), function ($definition) {
      return in_array($definition->getType(), self::MEDIAFLOW_FIELD_TYPES);
    });
    if (!count($reference_fields)) {
      return [];
    }

    $targets = [];
    foreach ($reference_fields as $definition) {
      $name = $definition->getName();
      if ($entity->hasField($name)) {
        /** @var \Drupal\Core\Field\FieldItemList $field_item */
        $field_item = $entity->get($name);
        array_push($targets, ...$field_item->getValue());
      }
    }
    return $targets;
  }

  /**
   * Finds any media embedded in wysiwyg fields in this entity.
   *
   * @param \Drupal\Core\Entity\FieldableEntityInterface $entity
   *   The entity to discover.
   *
   * @return \Drupal\media\MediaInterface[]
   *   List of media entities.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function getEmbeddedMediaEntities(FieldableEntityInterface $entity) {
    // Find field that may have embedded media.
    $fields = array_filter($entity->getFieldDefinitions(), function ($definition) {
      return in_array($definition->getType(), self::MEDIA_EMBED_FIELD_TYPES);
    });

    if (!count($fields)) {
      return [];
    }

    // Use regular expressions to search for embedded media.
    $uuids = [];
    foreach ($fields as $definition) {
      $name = $definition->getName();
      if ($entity->hasField($name)) {
        /** @var \Drupal\text\Plugin\Field\FieldType\TextFieldItemList $field_item */
        $field_item = $entity->get($name);
        $texts = array_column($field_item->getValue(), 'value');
        $text = implode($texts);
        $matches = [];
        preg_match_all('/.*<drupal-media.*data-entity-uuid="([a-zA-Z0-9\-]+)".*/', $text, $matches);
        foreach ($matches[1] as $match) {
          $uuids[] = $match;
        }
      }
    }
    // Load and return any found media entities.
    if (!count($uuids)) {
      return [];
    }

    $storage = $this->entityTypeManager->getStorage('media');
    $query = $storage->getQuery();
    $query->condition('uuid', $uuids, 'IN');
    $ids = $query->accessCheck()->execute();

    if (!count($ids)) {
      return [];
    }

    return $storage->loadMultiple($ids);
  }

  /**
   * Get ID form Mediaflow table for the given media entity.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity to get Mediaflow ID for.
   *
   * @return int|null
   *   The Mediaflow table ID.
   */
  public function getMediaflowIdFromMedia($media) {
    $source = $media->getSource();
    $fid = $source->getSourceFieldValue($media);

    $values = $this->database->select('mediaflow', 'fm')
      ->fields('fm', ['id'])
      ->condition('fm.file', $fid)
      ->execute()
      ->fetchCol();

    if (isset($values[0])) {
      return $values[0];
    }

    return NULL;
  }

}
