<?php

declare(strict_types=1);

namespace Drupal\meeting_api\Form;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Entity\BundleEntityFormBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\meeting_api\Entity\MeetingType;
use Drupal\meeting_api\ServerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for meeting type forms.
 */
final class MeetingTypeForm extends BundleEntityFormBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->entityTypeManager = $container->get('entity_type.manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    $servers = $this->entityTypeManager->getStorage('meeting_api_server')->loadMultiple();
    if (empty($servers)) {
      $form['no_servers_warning'] = [
        '#markup' => Xss::filter($this->t('There are no meeting servers available. Please <a href=":create_url">create one</a>.', [
          // This form is shown to privileged users. The route will check for
          // access for the user visiting, so we don't need to check it now.
          ':create_url' => Url::fromRoute('entity.meeting_api_server.add_form')->toString(),
        ])),
      ];

      return $form;
    }

    if ($this->operation === 'edit') {
      $form['#title'] = $this->t('Edit %label meeting type', ['%label' => $this->entity->label()]);
    }

    $form['label'] = [
      '#title' => $this->t('Label'),
      '#type' => 'textfield',
      '#default_value' => $this->entity->label(),
      '#description' => $this->t('The human-readable name of this meeting type.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $this->entity->id(),
      '#maxlength' => EntityTypeInterface::BUNDLE_MAX_LENGTH,
      '#machine_name' => [
        'exists' => [MeetingType::class, 'load'],
        'source' => ['label'],
      ],
      '#description' => $this->t('A unique machine-readable name for this meeting type. It must only contain lowercase letters, numbers, and underscores.'),
    ];

    $form['description'] = [
      '#title' => $this->t('Description'),
      '#description' => $this->t('The description of the meeting type.'),
      '#type' => 'textarea',
      '#default_value' => $this->entity->get('description'),
    ];

    $form['server_id'] = [
      '#type' => 'radios',
      '#title' => $this->t('Server'),
      '#description' => $this->t('The server to use for this meeting type.<br><em>Once saved, this value cannot be changed in this form. Consult the README.md for more information.</em>'),
      '#required' => TRUE,
      '#options' => array_map(static fn (ServerInterface $entity) => $entity->label(), $servers),
      '#default_value' => $this->entity->get('server_id'),
      '#disabled' => !$this->entity->isNew(),
    ];

    return $this->protectBundleIdElement($form);
  }

  /**
   * {@inheritdoc}
   */
  protected function copyFormValuesToEntity(EntityInterface $entity, array $form, FormStateInterface $form_state) {
    // Empty server won't stop the value to be copied to the entity.
    // Since the field is required the build will crash if we don't unset it.
    if ($form_state->getValue('server_id') === NULL) {
      $form_state->unsetValue('server_id');
    }
    parent::copyFormValuesToEntity($entity, $form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state): array {
    $actions = parent::actions($form, $form_state);
    $actions['submit']['#value'] = $this->t('Save meeting type');
    $actions['delete']['#value'] = $this->t('Delete meeting type');

    if (isset($form['no_servers_warning'])) {
      $actions['submit']['#access'] = FALSE;
    }

    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);

    $message_args = ['%label' => $this->entity->label()];
    $this->messenger()->addStatus(
      match($result) {
        SAVED_NEW => $this->t('The meeting type %label has been added.', $message_args),
        SAVED_UPDATED => $this->t('The meeting type %label has been updated.', $message_args),
      }
    );
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));

    return $result;
  }

}
