<?php

declare(strict_types=1);

namespace Drupal\Tests\meeting_api_manual\Functional;

use Drupal\meeting_api\Entity\MeetingType;
use Drupal\meeting_api\Entity\Server;
use Drupal\meeting_api\MeetingAttendee;
use Drupal\meeting_api\MeetingEntityInterface;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\meeting_api\Traits\MeetingFormTrait;
use Drupal\user\Entity\User;

/**
 * Tests the manual plugin.
 *
 * @group meeting_api_manual
 */
class PluginTest extends BrowserTestBase {

  use MeetingFormTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'meeting_api_manual',
  ];

  /**
   * Tests that the plugin offers no server configuration.
   */
  public function testNoServerConfiguration(): void {
    $assert_session = $this->assertSession();
    $user = $this->createUser([
      'access administration pages',
      'administer meeting_api_server',
      'administer site configuration',
      'view the administration theme',
    ]);
    $this->drupalLogin($user);

    $this->drupalGet('admin/config/services/meeting-api-server/add');
    $assert_session->fieldExists('Label')->setValue('Manual server');
    $assert_session->fieldExists('Machine-readable name')->setValue('manual_server');
    $assert_session->fieldExists('Backend')->selectOption('Manual URL');
    $assert_session->buttonExists('Change backend')->press();
    $assert_session->elementNotExists('css', 'details#meeting-api-backend-config-wrapper');
    $assert_session->buttonExists('Save')->click();
    $assert_session->statusMessageContains('Created new server Manual server.', 'status');
  }

  /**
   * Tests the meeting configuration form.
   */
  public function testMeetingConfiguration(): void {
    Server::create([
      'id' => 'manual_server',
      'label' => 'Manual server',
      'backend' => 'manual',
    ])->save();

    MeetingType::create([
      'id' => 'example',
      'label' => 'Example',
      'server_id' => 'manual_server',
    ])->save();

    $this->drupalLogin($this->drupalCreateUser([
      'administer meeting_api_meeting entities',
    ]));

    $this->drupalGet('/meeting/add/example');
    $label = $this->fillMeetingMandatoryFields();
    $assert_session = $this->assertSession();
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('Meeting URL field is required.', 'error');

    $assert_session->fieldExists('Meeting URL')->setValue('example');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('The URL example is not valid.', 'error');

    $assert_session->fieldExists('Meeting URL')->setValue('https://example.com');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains("New meeting {$label} has been created.", 'status');

    $meeting_storage = \Drupal::entityTypeManager()->getStorage('meeting_api_meeting');
    $meeting = array_values($meeting_storage->loadByProperties(['label' => $label]))[0];
    $this->assertInstanceOf(MeetingEntityInterface::class, $meeting);
    $this->assertEquals([
      'url' => 'https://example.com',
    ], $meeting->getSettings());

    // Edit the meeting to check that the value is re-populated correctly.
    $this->drupalGet($meeting->toUrl('edit-form'));
    $assert_session->fieldValueEquals('Meeting URL', 'https://example.com');

    /** @var \Drupal\meeting_api\MeetingManagerInterface $meeting_manager */
    $meeting_manager = \Drupal::service('Drupal\meeting_api\MeetingManagerInterface');
    $this->assertEquals(
      'https://example.com',
      $meeting_manager->joinMeeting($meeting, new MeetingAttendee(User::getAnonymousUser(), 'NONE')),
    );
  }

}
