<?php

declare(strict_types=1);

namespace Drupal\Tests\meeting_api\Functional;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Datetime\Entity\DateFormat;
use Drupal\meeting_api\Entity\MeetingType;
use Drupal\meeting_api\Entity\Server;
use Drupal\meeting_api\MeetingEntityInterface;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\meeting_api\Traits\AssertOptionsTrait;
use Drupal\Tests\meeting_api\Traits\MeetingFormTrait;

/**
 * Tests the meeting entity form.
 *
 * @group meeting_api
 */
class MeetingEntityFormTest extends BrowserTestBase {

  use AssertOptionsTrait;
  use MeetingFormTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'meeting_api_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->drupalPlaceBlock('local_actions_block');
  }

  /**
   * Tests CRUD operations on the meeting entity type.
   */
  public function testCrudOperations(): void {
    Server::create([
      'id' => 'simple_server',
      'label' => 'Simple server',
      'backend' => 'simple_no_forms',
    ])->save();

    $user = $this->drupalCreateUser([
      'administer meeting_api_meeting entities',
    ]);
    $this->drupalLogin($user);
    $this->drupalGet('/admin/content/meeting');
    $assert_session = $this->assertSession();
    $assert_session->pageTextContains('There are no meetings yet.');
    // No bundles exist, so the link to create a meeting is not visible.
    $assert_session->linkNotExists('Add meeting');

    MeetingType::create([
      'id' => 'default',
      'label' => 'Default',
      'server_id' => 'simple_server',
    ])->save();

    $this->drupalGet('/admin/content/meeting');
    $assert_session->pageTextContains('There are no meetings yet.');
    $this->clickLink('Add meeting');

    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('Label field is required.', 'error');
    $assert_session->statusMessageContains('The Start date date is required.', 'error');
    $assert_session->statusMessageContains('The End date date is required.', 'error');

    $attendees_number_field = $assert_session->fieldExists('Number of attendees');
    $this->assertEquals('0', $attendees_number_field->getValue());
    // Check that the number of attendees cannot be a number lower than 0.
    $attendees_number_field->setValue('-1');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('Number of attendees must be higher than or equal to 0.', 'error');
    $attendees_number_field->setValue('10');

    $assert_session->fieldExists('datetime[0][value][date]')->setValue('2020-10-05');
    $assert_session->fieldExists('datetime[0][value][time]')->setValue('12:00:00');
    $assert_session->fieldExists('datetime[0][end_value][date]')->setValue('2020-10-07');
    $assert_session->fieldExists('datetime[0][end_value][time]')->setValue('18:00:00');
    $assert_session->fieldExists('datetime[0][timezone]')->setValue('Asia/Tokyo');

    // Assert the presence of author, revision and created fields.
    $this->assertEquals(
      sprintf('%s (%s)', $user->getAccountName(), $user->id()),
      $assert_session->fieldExists('Author')->getValue(),
    );
    $this->assertEquals(
      (new DrupalDateTime())->format(DateFormat::load('html_date')->getPattern()),
      $assert_session->fieldExists('created[0][value][date]')->getValue(),
    );
    // There is no point in testing the correct time.
    $this->assertNotEmpty($assert_session->fieldExists('created[0][value][time]')->getValue());
    $assert_session->fieldExists('Revision log message');
    $assert_session->checkboxChecked('Enabled');

    // Finally, set a label and save.
    $assert_session->fieldExists('Label')->setValue('Meeting one');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('New meeting Meeting one has been created.', 'status');

    $meeting_storage = \Drupal::entityTypeManager()->getStorage('meeting_api_meeting');
    $meetings = $meeting_storage->loadMultiple();
    $this->assertCount(1, $meetings);
    $meeting = current($meetings);
    $this->assertEquals('Meeting one', $meeting->label());
    // Dates are stored in UTC.
    $this->assertEquals('2020-10-05T03:00:00', $meeting->get('datetime')->first()->get('value')->getValue());
    $this->assertEquals('2020-10-07T09:00:00', $meeting->get('datetime')->first()->get('end_value')->getValue());
    $this->assertEquals('Asia/Tokyo', $meeting->get('datetime')->first()->get('timezone')->getValue());
    $this->assertTrue($meeting->get('status')->first()->get('value')->getCastedValue());

    // Edit the meeting.
    $this->drupalGet('/admin/content/meeting');
    $this->clickLink('Edit');
    $assert_session->fieldValueEquals('Label', 'Meeting one');
    $assert_session->fieldValueEquals('Number of attendees', '10');
    $assert_session->fieldValueEquals('datetime[0][value][date]', '2020-10-05');
    $assert_session->fieldValueEquals('datetime[0][value][time]', '12:00:00');
    $assert_session->fieldValueEquals('datetime[0][end_value][date]', '2020-10-07');
    $assert_session->fieldValueEquals('datetime[0][end_value][time]', '18:00:00');
    $assert_session->fieldValueEquals('datetime[0][timezone]', 'Asia/Tokyo');
    $assert_session->checkboxChecked('Enabled');
    $assert_session->fieldExists('Label')->setValue('Changed title');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('The meeting Changed title has been updated.', 'status');

    $meetings = $meeting_storage->loadMultiple();
    $this->assertCount(1, $meetings);
    $meeting = current($meetings);
    $this->assertEquals('Changed title', $meeting->label());

    // Delete the meeting.
    $this->drupalGet('/admin/content/meeting');
    $this->clickLink('Delete');
    $assert_session->buttonExists('Delete')->press();
    $assert_session->statusMessageContains('The meeting Changed title has been deleted.', 'status');
    $assert_session->pageTextContains('There are no meetings yet.');
    $this->assertEmpty($meeting_storage->loadMultiple());
  }

  /**
   * Tests the backend settings available in the meeting form.
   */
  public function testBackendSettings(): void {
    // Create three servers and one meeting type for each of them.
    Server::create([
      'id' => 'foo_server',
      'label' => 'Foo server',
      'backend' => 'foo',
    ])->save();
    Server::create([
      'id' => 'bar_server',
      'label' => 'Bar server',
      'backend' => 'bar',
    ])->save();
    Server::create([
      'id' => 'acme_server',
      'label' => 'Acme server',
      'backend' => 'acme',
      'backend_config' => [
        'mode' => 'megazord',
      ],
    ])->save();

    MeetingType::create([
      'id' => 'foo',
      'label' => 'Foo',
      'server_id' => 'foo_server',
    ])->save();
    MeetingType::create([
      'id' => 'bar',
      'label' => 'Bar',
      'server_id' => 'bar_server',
    ])->save();
    MeetingType::create([
      'id' => 'acme',
      'label' => 'Acme',
      'server_id' => 'acme_server',
    ])->save();

    $this->drupalLogin($this->drupalCreateUser([
      'administer meeting_api_meeting entities',
    ]));

    // The "foo" backend plugin does not have dedicated settings for meetings.
    $this->drupalGet('/meeting/add/foo');
    $assert_session = $this->assertSession();
    $wrapper_selector = '[data-drupal-selector="edit-backend-settings-wrapper"]';

    // The "bar" backend plugin does implement a dedicated form for
    // meeting-specific configuration, but no server-specific configuration.
    // This test ensures that the two configuration forms are independent.
    $this->drupalGet('/meeting/add/bar');
    $wrapper = $assert_session->elementExists('css', $wrapper_selector);
    $this->assertRadiosOptions(
      $wrapper->find('named', ['fieldset', 'Mode']),
      ['One', 'Two', 'Three'],
    );
    $wrapper->selectFieldOption('backend_settings[mode]', 'two');
    $assert_session->fieldExists('Ignore this field', $wrapper)->setValue('I should not write here.');
    // Fill the mandatory fields.
    $label = $this->fillMeetingMandatoryFields();
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageNotExists('status');
    $assert_session->statusMessageContains('Ignore field should be kept empty.', 'error');
    $assert_session->fieldExists('Ignore this field', $wrapper)->setValue('');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains("New meeting $label has been created.", 'status');

    $meeting_storage = \Drupal::entityTypeManager()->getStorage('meeting_api_meeting');
    $meeting = array_values($meeting_storage->loadByProperties(['label' => $label]))[0];
    $this->assertInstanceOf(MeetingEntityInterface::class, $meeting);
    $this->assertEquals([
      'mode' => 'two',
    ], $meeting->get('backend_settings')->first()->getValue());

    // Edit the meeting and check that values are populated correctly.
    $this->drupalGet($meeting->toUrl('edit-form'));
    $this->assertTrue($wrapper->find('named', ['radio', 'Two'])->isChecked());

    // Test now the "acme" plugin, which implements both forms.
    $this->drupalGet('/meeting/add/acme');
    $this->assertCount(1, $wrapper->findAll('css', 'input'));
    $record_checkbox = $assert_session->elementExists('named', ['checkbox', 'Record meeting']);
    $this->assertFalse($record_checkbox->isChecked());
    $record_checkbox->check();
    // Check that the plugin is correctly passed to the meeting form, so that
    // developers can react to server settings.
    $this->assertEquals(
      'megazord',
      $assert_session->elementExists('css', '[data-test-attribute="mode-set"]')->getText(),
    );
    // Test that submit callbacks are called.
    $label = $this->fillMeetingMandatoryFields();
    $assert_session->buttonExists('Save')->press();
    $meeting = array_values($meeting_storage->loadByProperties(['label' => $label]))[0];
    $this->assertEquals([
      'record' => TRUE,
    ], $meeting->get('backend_settings')->first()->getValue());
  }

}
