<?php

declare(strict_types=1);

namespace Drupal\Tests\meeting_api\Functional;

use Drupal\meeting_api\Entity\Server;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\meeting_api\Traits\AssertOptionsTrait;

/**
 * Tests the meeting type config entity form.
 *
 * @group meeting_api
 */
class MeetingTypeFormTest extends BrowserTestBase {

  use AssertOptionsTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'meeting_api_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->drupalPlaceBlock('local_actions_block');
  }

  /**
   * Tests CRUD operations on the meeting type entity.
   */
  public function testCrudOperations(): void {
    $user = $this->drupalCreateUser([
      'administer meeting_api_meeting types',
    ]);
    $this->drupalLogin($user);

    $this->drupalGet('/admin/structure/meeting_api_meeting_types');
    $assert_session = $this->assertSession();
    $assert_session->pageTextContains('There are no meetings types yet.');
    $this->clickLink('Add meeting type');
    $assert_session->pageTextContains('There are no meeting servers available. Please create one.');

    Server::create([
      'id' => 'foo_server',
      'label' => 'Foo server',
      'backend' => 'foo',
    ])->save();

    $this->drupalGet('/admin/structure/meeting_api_meeting_types/add');
    $assert_session->buttonExists('Save meeting type')->press();
    $assert_session->statusMessageContains('Label field is required.', 'error');
    $assert_session->statusMessageContains('Machine-readable name field is required.', 'error');
    $assert_session->statusMessageContains('Server field is required.', 'error');

    $assert_session->fieldExists('Label')->setValue('Standard');
    $assert_session->fieldExists('Machine-readable name')->setValue('standard');
    $assert_session->fieldExists('Description')->setValue('Detailed description about the server.');

    $server_fieldset = $assert_session->elementExists('named', ['fieldset', 'Server']);
    $assert_session->pageTextContains('The server to use for this meeting type.');
    $assert_session->pageTextContains('Once saved, this value cannot be changed in this form. Consult the README.md for more information.');
    // Check that there's only one radio with the server label as label.
    $this->assertRadiosOptions($server_fieldset, ['Foo server']);
    // Select the above radio.
    $server_fieldset->selectFieldOption('server_id', 'foo_server');
    $assert_session->buttonExists('Save meeting type')->press();
    $assert_session->statusMessageContains('The meeting type Standard has been added.');
    $assert_session->pageTextNotContains('There are no meetings types yet.');
    $assert_session->elementExists('xpath', '//td[text()="Standard"]');

    $type_storage = \Drupal::entityTypeManager()->getStorage('meeting_api_meeting_type');
    $types = $type_storage->loadMultiple();
    $this->assertCount(1, $types);
    /** @var \Drupal\meeting_api\Entity\MeetingType $type */
    $type = current($types);
    $this->assertEquals('standard', $type->id());
    $this->assertEquals('Standard', $type->label());
    $this->assertEquals('Detailed description about the server.', $type->get('description'));
    $this->assertEquals('foo_server', $type->get('server_id'));

    $this->clickLink('Edit');
    $assert_session->fieldValueEquals('Label', 'Standard');
    $assert_session->fieldValueEquals('Description', 'Detailed description about the server.');
    $assert_session->fieldValueEquals('Machine-readable name', 'standard');
    $this->assertNotEmpty($assert_session->fieldExists('Machine-readable name')->getAttribute('disabled'));
    $this->assertNotEmpty($server_fieldset->find('named', ['radio', 'Foo server'])->getAttribute('disabled'));
    $this->assertTrue($server_fieldset->find('named', ['radio', 'Foo server'])->isChecked());

    Server::create([
      'id' => 'server_two',
      'label' => 'Another server for foo',
      'backend' => 'foo',
    ])->save();

    $this->drupalGet('/admin/structure/meeting_api_meeting_types/manage/standard');
    $this->assertRadiosOptions($server_fieldset, [
      'Foo server',
      'Another server for foo',
    ]);
    // Server cannot be changed anymore.
    $this->assertNotEmpty($server_fieldset->find('named', ['radio', 'Foo server'])->getAttribute('disabled'));
    $this->assertNotEmpty($server_fieldset->find('named', ['radio', 'Another server for foo'])->getAttribute('disabled'));
    $assert_session->fieldExists('Label')->setValue('Updated standard');
    $assert_session->buttonExists('Save meeting type')->press();
    $assert_session->statusMessageContains('The meeting type Updated standard has been updated.', 'status');

    $types = $type_storage->loadMultiple();
    $this->assertCount(1, $types);
    $type = $types['standard'];
    $this->assertEquals('Updated standard', $type->label());
    $this->assertEquals('Detailed description about the server.', $type->get('description'));
    $this->assertEquals('foo_server', $type->get('server_id'));

    // Test with a server of another type.
    Server::create([
      'id' => 'acme_server',
      'label' => 'Just another server',
      'backend' => 'acme',
    ])->save();

    $this->drupalGet('/admin/structure/meeting_api_meeting_types/add');
    $assert_session->fieldExists('Label')->setValue('Another meeting type');
    $assert_session->fieldExists('Machine-readable name')->setValue('another_meeting_type');
    $this->assertRadiosOptions($server_fieldset, [
      'Foo server',
      'Another server for foo',
      'Just another server',
    ]);
    $server_fieldset->selectFieldOption('server_id', 'acme_server');
    $assert_session->buttonExists('Save meeting type')->press();
    $assert_session->statusMessageContains('The meeting type Another meeting type has been added.', 'status');

    $types = $type_storage->loadMultiple();
    $this->assertCount(2, $types);
    $type = $types['another_meeting_type'];
    $this->assertEquals('Another meeting type', $type->label());
    $this->assertEmpty($type->get('description'));
    $this->assertEquals('acme_server', $type->get('server_id'));

    // Delete one of the types.
    $this->drupalGet('/admin/structure/meeting_api_meeting_types');
    $this->clickLink('Delete');
    $assert_session->buttonExists('Delete')->press();
    $assert_session->statusMessageContains('The meeting type Another meeting type has been deleted.', 'status');
    $assert_session->elementNotExists('xpath', '//table//*[text()="Another meeting type"]');
    $assert_session->elementExists('xpath', '//table//*[text()="Updated standard"]');

    $this->assertCount(1, $type_storage->loadMultiple());
  }

}
