<?php

declare(strict_types=1);

namespace Drupal\Tests\meeting_api\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\meeting_api\Entity\MeetingType;
use Drupal\meeting_api\Entity\Server;

/**
 * Class to test configuration related features.
 *
 * @group meeting_api
 */
class ConfigurationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'meeting_api',
    'meeting_api_test',
    'meeting_api_foobar_test',
  ];

  /**
   * Test server meant to use 'foo' plugin.
   */
  protected Server $fooServer;

  /**
   * Test server meant to use 'foobar' plugin.
   */
  protected Server $foobarServer;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('meeting_api_server');

    $this->fooServer = Server::create([
      'id' => 'fooServer',
      'label' => 'Foo server',
      'backend' => 'foo',
    ]);
    $this->fooServer->save();

    $this->foobarServer = Server::create([
      'id' => 'foobarServer',
      'label' => 'Bar server',
      'backend' => 'foobar',
    ]);
    $this->foobarServer->save();
  }

  /**
   * Tests server configuration dependencies.
   */
  public function testServerDependencies(): void {
    // Servers have same dependencies as plugins (module provider by default).
    $this->assertEquals(
      [
        'module' => [
          'meeting_api_test',
        ],
      ],
      $this->fooServer->getDependencies(),
    );
    $this->assertEquals(
      [
        'module' => [
          'meeting_api_foobar_test',
        ],
      ],
      $this->foobarServer->getDependencies(),
    );
  }

  /**
   * Tests meeting type configuration dependencies.
   */
  public function testMeetingTypeDependencies() {
    // Meeting types with each of the servers previously created.
    $meeting_type_foo = MeetingType::create([
      'id' => 'foo_type',
      'label' => 'Foo type',
      'server_id' => 'fooServer',
    ]);
    $meeting_type_foo->save();

    $meeting_type_foobar = MeetingType::create([
      'id' => 'foobar_type',
      'label' => 'Acme type',
      'server_id' => 'foobarServer',
    ]);
    $meeting_type_foobar->save();

    // Meeting types have a dependency on server entities.
    $this->assertEquals(
      [
        'config' => [
          $this->fooServer->getConfigDependencyName(),
        ],
      ],
      $meeting_type_foo->getDependencies(),
    );

    $this->assertEquals(
      [
        'config' => [
          $this->foobarServer->getConfigDependencyName(),
        ],
      ],
      $meeting_type_foobar->getDependencies(),
    );
  }

}
