<?php

declare(strict_types=1);

namespace Drupal\meeting_api;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\meeting_api\Exception\OperationException;

/**
 * Service to manage meetings.
 */
final class MeetingManager implements MeetingManagerInterface {

  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly BackendPluginManager $backendPluginManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function joinMeeting(MeetingInterface $meeting, MeetingAttendeeInterface $user): string {
    $server_id = $meeting->getServerId();
    /** @var \Drupal\meeting_api\ServerInterface $server */
    $server = $this->entityTypeManager
      ->getStorage('meeting_api_server')
      ->load($server_id);

    // It's an edge case to get to this point with a configuration dependency
    // from the MeetingType (bundle) on a non-existing Server, but we add it
    // just in case.
    if ($server === NULL) {
      throw new OperationException(sprintf('Server entity %d not found.', $server_id));
    }

    $backend = $this->backendPluginManager->createInstance(
      $server->get('backend'),
      $server->get('backend_config'),
    );
    assert($backend instanceof BackendInterface);

    return $backend->joinMeeting($meeting, $user);
  }

}
