<?php

declare(strict_types=1);

namespace Drupal\Tests\meeting_api\Functional;

use Drupal\Core\Extension\ModuleInstallerInterface;
use Drupal\Core\Url;
use Drupal\meeting_api\Entity\Server;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\meeting_api\Traits\AssertOptionsTrait;

/**
 * Tests server entity form.
 *
 * @group meeting_api
 */
class ServerFormTest extends BrowserTestBase {

  use AssertOptionsTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'meeting_api_test',
  ];

  /**
   * Tests access to the server form.
   */
  public function testAccess(): void {
    Server::create([
      'id' => 'foo_server',
      'label' => 'Foo server',
      'uuid' => '1234',
      'backend' => 'foo',
    ])->save();

    $routes = [
      'entity.meeting_api_server.collection' => [],
      'entity.meeting_api_server.add_form' => [],
      'entity.meeting_api_server.edit_form' => ['meeting_api_server' => 'foo_server'],
      'entity.meeting_api_server.delete_form' => ['meeting_api_server' => 'foo_server'],
    ];

    // Users without permission and with permission.
    $user = $this->createUser();
    $authorized_user = $this->createUser([
      'administer meeting_api_server',
    ]);

    foreach ($routes as $route_name => $route_params) {
      $url = Url::fromRoute($route_name, $route_params);
      $this->assertFalse(
        $url->access($user),
        sprintf('Failed on %s route for user without permissions.', $route_name),
      );
      $this->assertTrue(
        $url->access($authorized_user),
        sprintf('Failed on %s route for user with permissions.', $route_name),
      );
    }
  }

  /**
   * Tests CRUD operations on the server.
   */
  public function testCrudOperations(): void {
    $assert_session = $this->assertSession();
    $user = $this->createUser([
      'access administration pages',
      'administer meeting_api_server',
      'administer site configuration',
      'view the administration theme',
    ]);
    $this->drupalLogin($user);

    // Create a server.
    $this->drupalGet('admin/config');
    $this->clickLink('Meeting API servers');
    $assert_session->pageTextContains('Server configuration');
    $assert_session->pageTextContains('There are no servers yet.');
    $server_storage = \Drupal::entityTypeManager()->getStorage('meeting_api_server');
    $this->assertCount(0, $server_storage->loadMultiple());

    // Try to save without data to trigger required fields errors.
    $this->drupalGet('admin/config/services/meeting-api-server/add');
    $assert_session->fieldValueEquals('Label', '');
    $assert_session->fieldValueEquals('Machine-readable name', '');
    $assert_session->checkboxChecked('Enabled');
    $assert_session->fieldValueEquals('Description', '');
    $assert_session->fieldValueEquals('Backend', '');
    $assert_session->buttonExists('Save')->click();
    $assert_session->statusMessageContains('Label field is required.', 'error');
    $assert_session->statusMessageContains('Machine-readable name field is required.', 'error');
    $assert_session->statusMessageContains('Backend field is required.', 'error');

    // Set values for the server.
    $assert_session->fieldExists('Label')->setValue('Foo server');
    $assert_session->fieldExists('Machine-readable name')->setValue('foo_server');
    $assert_session->fieldExists('Enabled')->uncheck();
    $assert_session->fieldExists('Description')->setValue('Detailed description about the server.');
    $assert_session->optionExists('Backend', 'Acme');
    $assert_session->pageTextContains('The backend to use for this server');
    $assert_session->pageTextContains('Once saved, this value cannot be changed in this form. Consult the README.md for more information.');
    // The backend can be changed during creation.
    $this->assertSelectOptions([
      '- Select -' => '- Select -',
      'simple_no_forms' => 'Simple plugin without forms',
      'acme' => 'Acme',
      'bar' => 'Bar',
      'foo' => 'Foo',
    ], 'Backend');
    $assert_session->fieldExists('Backend')->selectOption('Foo');
    // When no JS is used, and the backend is changed, the user is required
    // to press the "Change backend" button.
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('You must configure the selected backend. Please press the "Change backend" button.', 'error');
    $assert_session->statusMessageNotExists('status');
    $assert_session->pageTextNotContains('Configure Foo backend');

    $assert_session->buttonExists('Change backend')->press();

    // Plugin sub-form fields will trigger errors too.
    $assert_session->pageTextContains('Configure Foo backend');
    $assert_session->fieldValueEquals('Foo URL', '');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('Foo URL field is required.', 'error');

    // Set values for the backend and save.
    $assert_session->fieldExists('Foo URL')->setValue('http://www.fooserver.com/');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('Created new server Foo server.', 'status');

    $servers = $server_storage->loadMultiple();
    $this->assertCount(1, $servers);
    /** @var \Drupal\meeting_api\Entity\Server $server */
    $server = current($servers);
    $server_values = array_diff_key($server->toArray(), array_flip(['uuid', 'langcode', 'dependencies']));
    $this->assertEquals(
      [
        'id' => 'foo_server',
        'label' => 'Foo server',
        'description' => 'Detailed description about the server.',
        'backend' => 'foo',
        'status' => FALSE,
        'backend_config' => [
          'url' => 'http://www.fooserver.com/',
        ],
      ],
      $server_values
    );

    // Edit the server.
    $this->drupalGet('admin/config/services/meeting-api-server/foo_server');
    $assert_session->fieldValueEquals('Label', 'Foo server');
    $assert_session->fieldValueEquals('Machine-readable name', 'foo_server');
    $assert_session->checkboxNotChecked('Enabled');
    $assert_session->fieldValueEquals('Description', 'Detailed description about the server.');
    $assert_session->fieldValueEquals('Backend', 'foo');
    $assert_session->fieldValueEquals('Foo URL', 'http://www.fooserver.com/');

    // The backend cannot be changed.
    $assert_session->fieldDisabled('Backend');
    $assert_session->buttonNotExists('Change backend');

    // The server and backend configuration can be changed.
    $assert_session->fieldExists('Label')->setValue('Foo server 2');
    $assert_session->pageTextContains('Configure Foo backend');
    $assert_session->fieldExists('Foo URL')->setValue('http://www.serverfoo.com/');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('Updated server Foo server 2.', 'status');

    $servers = $server_storage->loadMultiple();
    $this->assertCount(1, $servers);
    /** @var \Drupal\meeting_api\Entity\Server $server */
    $server = $servers['foo_server'];
    $server_values = array_diff_key($server->toArray(), array_flip(['uuid', 'langcode', 'dependencies']));
    $this->assertEquals(
      [
        'id' => 'foo_server',
        'label' => 'Foo server 2',
        'description' => 'Detailed description about the server.',
        'backend' => 'foo',
        'status' => FALSE,
        'backend_config' => [
          'url' => 'http://www.serverfoo.com/',
        ],
      ],
      $server_values
    );

    // The values are updated.
    $this->drupalGet('admin/config/services/meeting-api-server/foo_server');
    $assert_session->fieldValueEquals('Backend', 'foo');
    $assert_session->fieldValueEquals('Foo URL', 'http://www.serverfoo.com/');

    // Delete server.
    $this->clickLink('Delete');
    // Confirm.
    $assert_session->buttonExists('Delete')->press();
    $assert_session->pageTextContains('There are no servers yet.');
    $this->assertCount(0, $server_storage->loadMultiple());

    // Create another server. This time, change the backend during creation.
    $this->drupalGet('admin/config/services/meeting-api-server/add');
    $assert_session->fieldExists('Label')->setValue('Foo server');
    $assert_session->fieldExists('Machine-readable name')->setValue('foo_server');
    $assert_session->fieldExists('Enabled')->uncheck();
    $assert_session->fieldExists('Description')->setValue('Detailed description about the server.');
    $assert_session->fieldExists('Backend')->selectOption('Foo');
    $assert_session->buttonExists('Change backend')->press();
    $assert_session->fieldExists('Foo URL')->setValue('http://www.fooserver.com/');
    // Choose a different backend.
    $assert_session->fieldExists('Backend')->selectOption('Acme');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('You must configure the selected backend. Please press the "Change backend" button.', 'error');
    $assert_session->buttonExists('Change backend')->press();
    $assert_session->pageTextContains('Configure Acme backend');
    $assert_session->fieldValueEquals('Acme URL', '');
    $assert_session->fieldValueEquals('Acme Key', '');
    $assert_session->fieldValueEquals('Acme mode', 'Megazord');
    $assert_session->fieldNotExists('Foo URL');

    // Values are kept if the user tries to change the backend without selecting
    // a different backend.
    $assert_session->fieldValueEquals('Backend', 'Acme');
    $assert_session->fieldExists('Acme URL')->setValue('http://www.megazordserver.com/');
    $assert_session->fieldExists('Acme Key')->setValue('newacmeserverkey');
    $assert_session->fieldExists('Acme mode')->selectOption('Megazord');
    $assert_session->buttonExists('Change backend')->press();
    $assert_session->fieldValueEquals('Acme URL', 'http://www.megazordserver.com/');
    $assert_session->fieldValueEquals('Acme Key', 'newacmeserverkey');
    $assert_session->fieldValueEquals('Acme mode', 'Megazord');

    // The values are lost when switching between backends.
    $assert_session->fieldExists('Backend')->selectOption('Foo');
    $assert_session->buttonExists('Change backend')->press();
    $assert_session->fieldExists('Backend')->selectOption('Acme');
    $assert_session->buttonExists('Change backend')->press();
    $assert_session->pageTextContains('Configure Acme backend');
    $assert_session->fieldValueEquals('Acme URL', '');
    $assert_session->fieldValueEquals('Acme Key', '');
    $assert_session->fieldValueEquals('Acme mode', 'Megazord');
  }

  /**
   * Tests the form behavior if no backend plugins exist.
   */
  public function testNoBackends(): void {
    $user = $this->createUser([
      'access administration pages',
      'administer meeting_api_server',
      'administer site configuration',
      'view the administration theme',
    ]);
    $this->drupalLogin($user);
    $assert_session = $this->assertSession();

    // Attempt to create a server, while the plugin is present.
    $this->drupalGet('admin/config/services/meeting-api-server/add');
    $assert_session->buttonExists('Save');

    // Uninstall the module that defines the backend plugin.
    // @todo This will no longer work once the dependency detection is properly
    //   implemented. At that point, use a different mechanism, e.g. the hook,
    //   to let all plugins disappear.
    /** @var \Drupal\Core\Extension\ModuleInstallerInterface $module_installer */
    $module_installer = $this->container->get(ModuleInstallerInterface::class);
    $module_installer->uninstall(['meeting_api_test']);

    // The plugins are missing.
    $this->drupalGet('admin/config/services/meeting-api-server/add');
    $assert_session->pageTextContains('There are no meeting backend plugins available. Please install or create a module that defines meeting backend plugins.');
    // The regular form elements and the save button are not displayed.
    $assert_session->fieldNotExists('Label');
    $assert_session->fieldNotExists('Machine-readable name');
    $assert_session->fieldNotExists('Enabled');
    $assert_session->buttonNotExists('Save');
    // The delete button is still shown on edit forms.
    $assert_session->linkNotExists('Delete');

    // Reinstall the module that defines the backend plugin.
    $module_installer->install(['meeting_api_test']);

    // The plugins are back.
    $this->drupalGet('admin/config/services/meeting-api-server/add');
    $assert_session->buttonExists('Save');
  }

}
