<?php

declare(strict_types=1);

namespace Drupal\Tests\meeting_api\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\meeting_api\Traits\AssertOptionsTrait;

/**
 * Tests javascript functionality in server entity form.
 *
 * Especially, this tests how the sub-form for backend settings is replaced when
 * a different backend plugin is chosen.
 *
 * @group meeting_api
 */
class ServerFormAjaxTest extends WebDriverTestBase {

  use AssertOptionsTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'meeting_api_test',
  ];

  /**
   * Test the backend server configuration.
   */
  public function testBackendConfiguration(): void {
    $assert_session = $this->assertSession();
    $user = $this->createUser([
      'access administration pages',
      'administer meeting_api_server',
      'administer site configuration',
      'view the administration theme',
    ]);
    $this->drupalLogin($user);

    $this->drupalGet('admin/config/services/meeting-api-server/add');
    $assert_session->fieldExists('Label')->setValue('Foo server');
    $assert_session->fieldExists('Description')->setValue('Foo server description');
    $this->assertSelectOptions([
      '- Select -' => '- Select -',
      'simple_no_forms' => 'Simple plugin without forms',
      'acme' => 'Acme',
      'bar' => 'Bar',
      'foo' => 'Foo',
    ], 'Backend');

    $assert_session->fieldExists('Backend')->selectOption('Foo');
    $assert_session->assertWaitOnAjaxRequest();
    // The backend settings subform appears, with empty values.
    // Other form elements remain unchanged.
    $assert_session->pageTextContains('Configure Foo backend');
    $assert_session->fieldValueEquals('Foo URL', '');
    $assert_session->fieldExists('Foo URL')->setValue('http://www.fooserver.com/');
    $assert_session->fieldValueEquals('Label', 'Foo server');
    $assert_session->fieldValueEquals('Description', 'Foo server description');

    // Switching the backend resets the backend configuration.
    $assert_session->fieldExists('Backend')->selectOption('Acme');
    $assert_session->assertWaitOnAjaxRequest();
    $assert_session->fieldValueEquals('Label', 'Foo server');
    $assert_session->fieldValueEquals('Description', 'Foo server description');
    $assert_session->pageTextContains('Configure Acme backend');
    $assert_session->fieldValueEquals('Acme URL', '');
    $assert_session->fieldValueEquals('Acme Key', '');
    $assert_session->fieldValueEquals('Acme mode', 'Megazord');
    $this->assertSelectOptions([
      'megazord' => 'Megazord',
      'gigazord' => 'Gigazord',
      'ultrazord' => 'Ultrazord',
    ], 'Acme mode');
    $assert_session->fieldNotExists('Foo URL');

    // Set values for the backend.
    $assert_session->fieldExists('Acme URL')->setValue('http://www.acmeserver.com/');
    $assert_session->fieldExists('Acme Key')->setValue('acmeserverkey');
    $assert_session->fieldExists('Acme mode')->selectOption('Gigazord');

    // Switch to the previous backend, the values are reset.
    $assert_session->fieldExists('Backend')->selectOption('Foo');
    $assert_session->assertWaitOnAjaxRequest();
    $assert_session->fieldValueEquals('Foo URL', '');
    $assert_session->fieldNotExists('Acme URL');
    $assert_session->fieldNotExists('Acme Key');
    $assert_session->fieldNotExists('Acme mode');

    // Set new values for the backend and save.
    $assert_session->fieldExists('Foo URL')->setValue('http://www.fooserver.com/');
    $assert_session->buttonExists('Save')->click();
    $assert_session->statusMessageContains('Created new server Foo server.', 'status');

    // Values are saved.
    $this->clickLink('Edit');
    $assert_session->fieldValueEquals('Label', 'Foo server');
    $assert_session->fieldValueEquals('Backend', 'foo');
    $assert_session->fieldValueEquals('Foo URL', 'http://www.fooserver.com/');
    // In an edit form, the backend cannot be changed.
    $assert_session->fieldDisabled('Backend');
  }

  /**
   * Tests that backend plugins without forms work correctly.
   */
  public function testBackendNoForm(): void {
    $assert_session = $this->assertSession();
    $user = $this->createUser([
      'access administration pages',
      'administer meeting_api_server',
      'administer site configuration',
      'view the administration theme',
    ]);
    $this->drupalLogin($user);

    $this->drupalGet('admin/config/services/meeting-api-server/add');
    $assert_session->fieldExists('Label')->setValue('Example server');
    $assert_session->fieldExists('Backend')->selectOption('Simple plugin without forms');
    $assert_session->assertWaitOnAjaxRequest();
    $assert_session->elementNotExists('css', 'details#meeting-api-backend-config-wrapper');
    $assert_session->buttonExists('Save')->click();
    $assert_session->statusMessageContains('Created new server Example server.', 'status');
  }

}
