<?php

declare(strict_types=1);

namespace Drupal\Tests\meeting_api\Unit;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\meeting_api\BackendInterface;
use Drupal\meeting_api\BackendPluginManager;
use Drupal\meeting_api\Exception\OperationException;
use Drupal\meeting_api\MeetingAttendeeInterface;
use Drupal\meeting_api\MeetingInterface;
use Drupal\meeting_api\MeetingManager;
use Drupal\meeting_api\ServerInterface;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the meeting manager.
 *
 * @coversDefaultClass \Drupal\meeting_api\MeetingManager
 *
 * @group meeting_api
 */
class MeetingManagerTest extends UnitTestCase {

  /**
   * Tests the joinMeeting method.
   *
   * @covers ::joinMeeting
   */
  public function testJoinMeeting(): void {
    $server = $this->createMock(ServerInterface::class);
    $server
      ->expects($this->exactly(2))
      ->method('get')
      ->willReturnCallback(function (string $property_name): mixed {
        return match($property_name) {
          'backend' => 'foo_plugin',
          'backend_config' => [
            'key1' => 'value1',
            'key2' => 'value2',
          ],
        };
      });

    $storage = $this->createMock(EntityStorageInterface::class);
    $storage
      ->expects($this->once())
      ->method('load')
      ->with($this->equalTo('test_server'))
      ->willReturn($server);

    $entity_type_manager = $this->createMock(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->expects($this->once())
      ->method('getStorage')
      ->with($this->equalTo('meeting_api_server'))
      ->willReturn($storage);

    $meeting = $this->createMock(MeetingInterface::class);
    $meeting
      ->expects($this->once())
      ->method('getServerId')
      ->willReturn('test_server');
    $attendee = $this->createStub(MeetingAttendeeInterface::class);

    $plugin = $this->createMock(BackendInterface::class);
    $plugin
      ->expects($this->once())
      ->method('joinMeeting')
      ->with(
        $this->equalTo($meeting),
        $this->equalTo($attendee),
      )
      ->willReturn('https://www.example.com');

    $plugin_manager = $this->createMock(BackendPluginManager::class);
    $plugin_manager
      ->expects($this->once())
      ->method('createInstance')
      ->with(
        $this->equalTo('foo_plugin'),
        $this->equalTo([
          'key1' => 'value1',
          'key2' => 'value2',
        ]),
      )
      ->willReturn($plugin);

    $meeting_manager = new MeetingManager($entity_type_manager, $plugin_manager);
    $this->assertEquals(
      'https://www.example.com',
      $meeting_manager->joinMeeting($meeting, $attendee),
    );
  }

  /**
   * Tests join meeting operation with missing server.
   *
   * @covers ::joinMeeting
   */
  public function testJoinMeetingMissingServer(): void {
    $storage = $this->createMock(EntityStorageInterface::class);
    $storage
      ->expects($this->once())
      ->method('load')
      ->with($this->equalTo('test_server'))
      ->willReturn(NULL);

    $entity_type_manager = $this->createMock(EntityTypeManagerInterface::class);
    $entity_type_manager
      ->expects($this->once())
      ->method('getStorage')
      ->with($this->equalTo('meeting_api_server'))
      ->willReturn($storage);

    $plugin_manager = $this->createMock(BackendPluginManager::class);
    $plugin_manager
      ->expects($this->never())
      ->method('createInstance');

    $meeting = $this->createMock(MeetingInterface::class);
    $meeting
      ->expects($this->once())
      ->method('getServerId')
      ->willReturn('test_server');

    $this->expectException(OperationException::class);
    $this->expectExceptionMessage('Server entity 0 not found.');

    $meeting_manager = new MeetingManager($entity_type_manager, $plugin_manager);
    $meeting_manager->joinMeeting(
      $meeting,
      $this->createStub(MeetingAttendeeInterface::class),
    );
  }

}
