<?php

declare(strict_types=1);

namespace Drupal\meeting_api\Entity;

use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\meeting_api\MeetingEntityInterface;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the meeting entity class.
 *
 * @ContentEntityType(
 *   id = "meeting_api_meeting",
 *   label = @Translation("Meeting"),
 *   label_collection = @Translation("Meetings"),
 *   label_singular = @Translation("meeting"),
 *   label_plural = @Translation("meetings"),
 *   label_count = @PluralTranslation(
 *     singular = "@count meeting",
 *     plural = "@count meetings",
 *   ),
 *   bundle_label = @Translation("Meeting type"),
 *   handlers = {
 *     "list_builder" = "Drupal\meeting_api\MeetingListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "add" = "Drupal\meeting_api\Form\MeetingForm",
 *       "edit" = "Drupal\meeting_api\Form\MeetingForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *       "revision-delete" = \Drupal\Core\Entity\Form\RevisionDeleteForm::class,
 *       "revision-revert" = \Drupal\Core\Entity\Form\RevisionRevertForm::class,
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\meeting_api\Routing\MeetingHtmlRouteProvider",
 *       "revision" = \Drupal\Core\Entity\Routing\RevisionHtmlRouteProvider::class,
 *     },
 *   },
 *   base_table = "meeting_api_meeting",
 *   revision_table = "meeting_api_meeting_revision",
 *   show_revision_ui = TRUE,
 *   admin_permission = "administer meeting_api_meeting entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "revision_id",
 *     "bundle" = "bundle",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_uid",
 *     "revision_created" = "revision_timestamp",
 *     "revision_log_message" = "revision_log",
 *   },
 *   links = {
 *     "collection" = "/admin/content/meeting",
 *     "add-form" = "/meeting/add/{meeting_api_meeting_type}",
 *     "add-page" = "/meeting/add",
 *     "canonical" = "/meeting/{meeting_api_meeting}",
 *     "edit-form" = "/meeting/{meeting_api_meeting}",
 *     "delete-form" = "/meeting/{meeting_api_meeting}/delete",
 *     "delete-multiple-form" = "/admin/content/meeting/delete-multiple",
 *     "revision" = "/meeting/{meeting_api_meeting}/revision/{meeting_api_meeting_revision}/view",
 *     "revision-delete-form" = "/meeting/{meeting_api_meeting}/revision/{meeting_api_meeting_revision}/delete",
 *     "revision-revert-form" = "/meeting/{meeting_api_meeting}/revision/{meeting_api_meeting_revision}/revert",
 *     "version-history" = "/meeting/{meeting_api_meeting}/revisions",
 *   },
 *   bundle_entity_type = "meeting_api_meeting_type",
 *   field_ui_base_route = "entity.meeting_api_meeting_type.edit_form",
 * )
 */
final class Meeting extends RevisionableContentEntityBase implements MeetingEntityInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function getIdentifier(): string {
    return $this->uuid() ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Label'))
      ->setRequired(TRUE)
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Status'))
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setDefaultValue(TRUE)
      ->setSetting('on_label', 'Enabled')
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => FALSE,
        ],
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'boolean',
        'label' => 'above',
        'weight' => 0,
        'settings' => [
          'format' => 'enabled-disabled',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Author'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getDefaultEntityOwner')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the meeting was created.'))
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setDescription(t('The time that the meeting was last edited.'));

    $fields['datetime'] = BaseFieldDefinition::create('daterange_timezone')
      ->setLabel(t('Start and end time'))
      ->setDescription(t('The start and end time of the meeting.'))
      ->setRequired(TRUE)
      ->setRevisionable(TRUE)
      ->setTranslatable(FALSE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'daterange_timezone',
        'weight' => 25,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'daterange_timezone',
        'weight' => 25,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['max_attendees'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Number of attendees'))
      ->setDescription(t('The maximum allowed number of attendees. Set to zero for unlimited.'))
      ->setRequired(TRUE)
      ->setRevisionable(TRUE)
      ->setTranslatable(FALSE)
      ->setDefaultValue(MeetingEntityInterface::ATTENDEES_UNLIMITED)
      ->setSetting('unsigned', TRUE)
      ->setSetting('min', 0)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_integer',
        'weight' => 30,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => 30,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['backend_settings'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Backend settings'))
      ->setRevisionable(TRUE)
      ->setRequired(TRUE)
      ->setDefaultValue([])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'meeting_api_backend_settings_widget',
        'weight' => 35,
      ])
      ->setDisplayConfigurable('view', FALSE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getServerId(): string {
    $bundle = $this->get('bundle')->first()->get('entity')->getValue();

    if ($bundle === NULL) {
      throw new \LogicException(sprintf('Missing bundle on entity, entity type meeting_api_meeting, entity id %s.', $this->id()));
    }

    return $bundle->get('server_id');
  }

  /**
   * {@inheritdoc}
   */
  public function getSettings(): array {
    return $this->get('backend_settings')->first()?->getValue() ?? [];
  }

}
