<?php

declare(strict_types=1);

namespace Drupal\meeting_api\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBundleBase;

/**
 * Defines the Meeting type configuration entity.
 *
 * @ConfigEntityType(
 *   id = "meeting_api_meeting_type",
 *   label = @Translation("Meeting type"),
 *   label_collection = @Translation("Meeting types"),
 *   label_singular = @Translation("meeting type"),
 *   label_plural = @Translation("meetings types"),
 *   label_count = @PluralTranslation(
 *     singular = "@count meetings type",
 *     plural = "@count meetings types",
 *   ),
 *   handlers = {
 *     "form" = {
 *       "add" = "Drupal\meeting_api\Form\MeetingTypeForm",
 *       "edit" = "Drupal\meeting_api\Form\MeetingTypeForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *     },
 *     "list_builder" = "Drupal\meeting_api\MeetingTypeListBuilder",
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   admin_permission = "administer meeting_api_meeting types",
 *   bundle_of = "meeting_api_meeting",
 *   config_prefix = "meeting_api_meeting_type",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *   },
 *   links = {
 *     "add-form" = "/admin/structure/meeting_api_meeting_types/add",
 *     "edit-form" = "/admin/structure/meeting_api_meeting_types/manage/{meeting_api_meeting_type}",
 *     "delete-form" = "/admin/structure/meeting_api_meeting_types/manage/{meeting_api_meeting_type}/delete",
 *     "collection" = "/admin/structure/meeting_api_meeting_types",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "description",
 *     "server_id",
 *     "uuid",
 *   },
 * )
 */
final class MeetingType extends ConfigEntityBundleBase {

  /**
   * The machine name of this meeting type.
   */
  protected string $id;

  /**
   * The human-readable name of the meeting type.
   */
  protected string $label;

  /**
   * The server ID for this meeting type.
   */
  protected string $server_id;

  /**
   * A non-mandatory description of the meeting type.
   *
   * @var string|null
   */
  protected ?string $description = NULL;

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    parent::calculateDependencies();

    $storage = $this->entityTypeManager()->getStorage('meeting_api_server');
    $server = $storage->load($this->server_id);
    $name = $server->getConfigDependencyName();

    $this->addDependency('config', $name);

    return $this;
  }

}
