<?php

declare(strict_types=1);

namespace Drupal\meeting_api\Entity;

use Drupal\Component\Plugin\LazyPluginCollection;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Plugin\DefaultSingleLazyPluginCollection;
use Drupal\meeting_api\BackendPluginManager;
use Drupal\meeting_api\ServerInterface;

/**
 * Defines the server entity type.
 *
 * @ConfigEntityType(
 *   id = "meeting_api_server",
 *   label = @Translation("Server"),
 *   label_collection = @Translation("Servers"),
 *   label_singular = @Translation("server"),
 *   label_plural = @Translation("servers"),
 *   label_count = @PluralTranslation(
 *     singular = "@count server",
 *     plural = "@count servers",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\meeting_api\ServerListBuilder",
 *     "form" = {
 *       "add" = "Drupal\meeting_api\Form\ServerForm",
 *       "edit" = "Drupal\meeting_api\Form\ServerForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *     },
 *   },
 *   config_prefix = "meeting_api_server",
 *   admin_permission = "administer meeting_api_server",
 *   links = {
 *     "collection" = "/admin/config/services/meeting-api-server",
 *     "add-form" = "/admin/config/services/meeting-api-server/add",
 *     "edit-form" = "/admin/config/services/meeting-api-server/{meeting_api_server}",
 *     "delete-form" = "/admin/config/services/meeting-api-server/{meeting_api_server}/delete",
 *   },
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "description",
 *     "backend",
 *     "backend_config",
 *   },
 * )
 */
class Server extends ConfigEntityBase implements ServerInterface {

  /**
   * The plugin collection that holds the backend plugin for this entity.
   */
  protected LazyPluginCollection $pluginCollection;

  /**
   * The server ID.
   */
  protected string $id;

  /**
   * The server label.
   */
  protected string $label;

  /**
   * The server description.
   */
  protected string $description = '';

  /**
   * The ID of the backend plugin.
   */
  protected string $backend;

  /**
   * The configuration of the backend plugin.
   */
  protected array $backend_config = [];

  /**
   * Retrieves the plugin collection related to backend plugin.
   *
   * @return \Drupal\Component\Plugin\LazyPluginCollection
   *   The plugin collection containing the backend plugin used by this server.
   */
  protected function getPluginCollection(): LazyPluginCollection {
    return $this->pluginCollection ??= new DefaultSingleLazyPluginCollection(
      \Drupal::service(BackendPluginManager::class),
      $this->get('backend'),
      $this->get('backend_config'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getPluginCollections(): array {
    if (!isset($this->backend)) {
      // The object is not fully initialized.
      // This happens when the server create form is submitted with no backend.
      return [];
    }

    return [
      'backend_config' => $this->getPluginCollection(),
    ];
  }

}
