<?php

declare(strict_types=1);

namespace Drupal\meeting_api\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\PluginFormFactoryInterface;
use Drupal\meeting_api\BackendInterface;
use Drupal\meeting_api\BackendPluginManager;
use Drupal\meeting_api\Entity\Server;
use Drush\Commands\AutowireTrait;

/**
 * Server form.
 */
class ServerForm extends EntityForm {

  use AutowireTrait;

  public function __construct(
    protected BackendPluginManager $backendPluginManager,
    protected PluginFormFactoryInterface $pluginFormFactory,
    MessengerInterface $messenger,
  ) {
    $this->setMessenger($messenger);
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    $backend_options = $this->getBackendOptions();

    if ($backend_options === []) {
      $form['no_backends_warning'] = [
        '#markup' => $this->t('There are no meeting backend plugins available. Please install or create a module that defines meeting backend plugins.'),
      ];

      return $form;
    }

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $this->entity->label(),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $this->entity->id(),
      '#machine_name' => [
        'exists' => [Server::class, 'load'],
      ],
      '#disabled' => !$this->entity->isNew(),
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $this->entity->get('description'),
    ];

    // Use the stored value for the backend. Use the entity value on first form
    // loads.
    // @see ::updateBackend()
    $backend_id = $form_state->get('backend') ?: $this->entity->get('backend');

    // Common settings for the select and related submit button.
    $ajax_settings = [
      'callback' => [$this, 'updateBackendAjax'],
      'wrapper' => 'meeting-api-backend-config-wrapper',
    ];

    $form['backend'] = [
      '#type' => 'select',
      '#title' => $this->t('Backend'),
      '#description' => $this->t('The backend to use for this server.<br><em>Once saved, this value cannot be changed in this form. Consult the README.md for more information.</em>'),
      '#options' => $backend_options,
      // @todo Check that saved value is a valid backend.
      '#default_value' => $backend_id,
      '#required' => TRUE,
      '#ajax' => $ajax_settings + [
        'trigger_as' => ['name' => 'backend_change'],
      ],
    ];
    $form['backend_change'] = [
      '#type' => 'submit',
      '#value' => $this->t('Change backend'),
      '#name' => 'backend_change',
      '#submit' => [[$this, 'updateBackend']],
      '#ajax' => $ajax_settings,
      '#limit_validation_errors' => [['backend']],
      '#attributes' => ['class' => ['js-hide']],
    ];

    if (!$this->entity->isNew()) {
      // Disable the element, but keep it visible for informational purposes.
      $form['backend']['#disabled'] = TRUE;
      // The button is not needed, and would look out of place with js disabled.
      $form['backend_change']['#access'] = FALSE;
    }

    // No backend selected by default.
    $form['backend_config'] = [
      '#type' => 'container',
      '#tree' => TRUE,
      '#attributes' => ['id' => 'meeting-api-backend-config-wrapper'],
    ];

    if (!empty($backend_id)) {
      // Make sure to store the backend in state, if a value is provided.
      // This is mostly useful on first form loads.
      $form_state->set('backend', $backend_id);

      // Instantiate the backend using the submitted form values, or fall back
      // to the entity values.
      $backend = $this->getBackend($backend_id, $form_state->getValue('backend_config', $this->entity->get('backend_config')));
      if ($backend->hasFormClass(BackendInterface::PLUGIN_FORM_CONFIGURE)) {
        // Attach the backend plugin configuration form.
        $backend_form_state = SubformState::createForSubform($form['backend_config'], $form, $form_state);
        $form['backend_config'] = $this->pluginFormFactory
          ->createInstance($backend, BackendInterface::PLUGIN_FORM_CONFIGURE)
          ->buildConfigurationForm($form['backend_config'], $backend_form_state);
        // Modify the backend plugin configuration container element.
        $form['backend_config']['#type'] = 'details';
        $form['backend_config']['#title'] = $this->t('Configure %plugin backend', ['%plugin' => $backend->label()]);
        $form['backend_config']['#open'] = TRUE;
      }
    }

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $this->entity->status(),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);

    if (isset($form['no_backends_warning'])) {
      // Hide the save button, but keep the delete button.
      $actions['submit']['#access'] = FALSE;
    }

    return $actions;
  }

  /**
   * Callback to switch the backend configuration.
   *
   * @param array $form
   *   The build form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function updateBackend(array $form, FormStateInterface $form_state): void {
    $submitted_backend_id = $form_state->getValue('backend');
    $current_backend_id = $form_state->get('backend');

    if ($submitted_backend_id !== $current_backend_id) {
      $form_state->set('backend', $submitted_backend_id);
      $form_state->setValue('backend_config', []);
      // Set the empty value for the backend configuration if changed.
      // Reset the configuration as if the user would have done.
      $user_input = $form_state->getUserInput();
      unset($user_input['backend_config']);
      $form_state->setUserInput($user_input);
    }

    $form_state->setRebuild();
  }

  /**
   * Ajax callback to update the backend configuration form element.
   *
   * @param array $form
   *   The build form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The backend configuration form element.
   */
  public function updateBackendAjax(array $form, FormStateInterface $form_state): array {
    return $form['backend_config'];
  }

  /**
   * Retrieves the available backends as options.
   *
   * @return array
   *   The backend options.
   */
  protected function getBackendOptions(): array {
    $backends = $this->backendPluginManager->getDefinitions();
    $backend_options = [];

    foreach ($backends as $backend_id => $definition) {
      $backend = $this->getBackend($backend_id);
      $backend_options[$backend_id] = $backend->label();
    }

    return $backend_options;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // If the "Change backend" button has been pressed, we don't need to run
    // validations.
    if ($form_state->getTriggeringElement()['#name'] === 'backend_change') {
      return;
    }

    // When no backend is selected, the default form validation will trigger.
    $submitted_backend_id = $form_state->getValue('backend');
    if ($submitted_backend_id === NULL) {
      return;
    }

    // If the submitted value differs from the one stored in the state, it
    // means that the submit callback of the change button was not executed.
    // This should happen only when the form is being used without JS.
    $current_backend_id = $form_state->get('backend');
    if ($submitted_backend_id !== $current_backend_id) {
      $form_state->setErrorByName('backend', $this->t('You must configure the selected backend. Please press the "@button" button.', [
        '@button' => $this->t('Change backend'),
      ]));
      return;
    }

    // Validate configuration otherwise.
    $backend = $this->getBackend($current_backend_id, $form_state->getValue('backend_config', []));
    if ($backend->hasFormClass(BackendInterface::PLUGIN_FORM_CONFIGURE)) {
      $backend_form_state = SubformState::createForSubform($form['backend_config'], $form, $form_state);
      $this->pluginFormFactory
        ->createInstance($backend, BackendInterface::PLUGIN_FORM_CONFIGURE)
        ->validateConfigurationForm($form['backend_config'], $backend_form_state);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): EntityInterface {
    parent::submitForm($form, $form_state);

    $backend = $this->getBackend($form_state->get('backend'), $form_state->getValue('backend_config', []));
    if ($backend->hasFormClass(BackendInterface::PLUGIN_FORM_CONFIGURE)) {
      $backend_form_state = SubformState::createForSubform($form['backend_config'], $form, $form_state);
      $this->pluginFormFactory
        ->createInstance($backend, BackendInterface::PLUGIN_FORM_CONFIGURE)
        ->submitConfigurationForm($form['backend_config'], $backend_form_state);
    }

    $this->entity->set('backend_config', $backend->getConfiguration());

    return $this->entity;
  }

  /**
   * {@inheritdoc}
   */
  protected function copyFormValuesToEntity(EntityInterface $entity, array $form, FormStateInterface $form_state) {
    // Avoid copying an empty backend to the entity during validation.
    if ($form_state->getValue('backend', '') === '') {
      $form_state->unsetValue('backend');
    }
    parent::copyFormValuesToEntity($entity, $form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $this->entity->label()];
    $this->messenger()->addStatus(
      match($result) {
        SAVED_NEW => $this->t('Created new server %label.', $message_args),
        SAVED_UPDATED => $this->t('Updated server %label.', $message_args),
      },
    );
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $result;
  }

  /**
   * Returns instantiated backend given a plugin ID and configuration.
   *
   * @param string $id
   *   The plugin ID.
   * @param array $config
   *   The plugin configuration array.
   *
   * @return \Drupal\meeting_api\BackendInterface
   *   The plugin object.
   */
  protected function getBackend(string $id, array $config = []): BackendInterface {
    // @todo This is not much needed now.
    return $this->backendPluginManager->createInstance($id, $config);
  }

}
