<?php

namespace Drupal\meeting_api\Plugin\Field\FieldWidget;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Field\WidgetInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\meeting_api\BackendInterface;
use Drupal\meeting_api\BackendPluginManager;
use Drupal\meeting_api\MeetingEntityInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'backend_settings' widget.
 */
#[FieldWidget(
  id: 'meeting_api_backend_settings_widget',
  label: new TranslatableMarkup('Backend Settings Widget'),
  field_types: ['map'],
  multiple_values: TRUE,
)]
class BackendSettingsWidget extends WidgetBase implements WidgetInterface, ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected PluginFormFactoryInterface $pluginFormFactory,
    protected BackendPluginManager $backendPluginManager,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get(EntityTypeManagerInterface::class),
      $container->get(PluginFormFactoryInterface::class),
      $container->get(BackendPluginManager::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $meeting = $items->getEntity();
    assert($meeting instanceof MeetingEntityInterface);
    $form_state->set('meeting_entity', $meeting);
    $backend = $this->getBackendInstance($meeting);
    if (!$backend->hasFormClass(BackendInterface::PLUGIN_FORM_MEETING)) {
      // Let the default value take place in case the Backend does not support
      // form.
      $element += [
        '#type' => 'hidden',
        '#value' => $this->fieldDefinition->getDefaultValue($meeting),
      ];

      return $element;
    }

    $element += [
      '#type' => 'container',
      '#tree' => TRUE,
      '#element_validate' => [[$this, 'validateForm']],
    ];

    // Attach the backend plugin configuration form.
    $backend_form_state = SubformState::createForSubform($element, $form, $form_state);
    $element = $this->pluginFormFactory
      ->createInstance($backend, BackendInterface::PLUGIN_FORM_MEETING)
      ->buildConfigurationForm($element, $backend_form_state);

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$element, FormStateInterface $form_state, array $form) {
    $meeting = $form_state->get('meeting_entity');
    if ($meeting === NULL) {
      return;
    }

    $backend = $this->getBackendInstance($meeting);
    if (!$backend->hasFormClass(BackendInterface::PLUGIN_FORM_MEETING)) {
      return;
    }

    $backend_form_state = SubformState::createForSubform($element, $form, $form_state);
    $this->pluginFormFactory
      ->createInstance($backend, BackendInterface::PLUGIN_FORM_MEETING)
      ->validateConfigurationForm($element, $backend_form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function extractFormValues(FieldItemListInterface $items, array $form, FormStateInterface $form_state) {
    $meeting = $form_state->get('meeting_entity');
    if ($meeting === NULL) {
      parent::extractFormValues($items, $form, $form_state);
      return;
    }

    $backend = $this->getBackendInstance($meeting);
    if (!$backend->hasFormClass(BackendInterface::PLUGIN_FORM_MEETING)) {
      parent::extractFormValues($items, $form, $form_state);
      return;
    }

    // Allow to modify the form_state values in the plugin submit.
    $field_name = $items->getName();
    $backend_form_state = SubformState::createForSubform($form[$field_name]['widget'], $form, $form_state);
    $this->pluginFormFactory
      ->createInstance($backend, BackendInterface::PLUGIN_FORM_MEETING)
      ->submitConfigurationForm($form[$field_name]['widget'], $backend_form_state);

    parent::extractFormValues($items, $form, $form_state);
  }

  /**
   * Returns an instance of the backend plugin for this meeting entity.
   *
   * The backend plugin is set in the meeting bundle entity.
   *
   * @param \Drupal\meeting_api\MeetingEntityInterface $meeting
   *   The meeting where to get the backend.
   *
   * @return \Drupal\meeting_api\BackendInterface
   *   The backend plugin instance.
   */
  protected function getBackendInstance(MeetingEntityInterface $meeting): BackendInterface {
    $server_id = $meeting->getServerId();
    $server = $this->entityTypeManager
      ->getStorage('meeting_api_server')
      ->load($server_id);

    if (!$server) {
      throw new \RuntimeException(sprintf('Server entity %d not found.', $server_id));
    }

    // @todo Maybe add methods for backend/backend_config and server_id above?
    //   And maybe uniform server_id/backend with both _id or both without.
    $backend = $this->backendPluginManager->createInstance($server->get('backend'), $server->get('backend_config'));
    assert($backend instanceof BackendInterface);

    return $backend;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    if (!$field_definition instanceof BaseFieldDefinition) {
      return FALSE;
    }

    $form_display_options = $field_definition->getDisplayOptions('form');
    if (!isset($form_display_options['type'])) {
      return FALSE;
    }

    return $form_display_options['type'] === 'meeting_api_backend_settings_widget';
  }

}
