<?php

declare(strict_types=1);

namespace Drupal\meeting_api_bbb\Plugin\MeetingApiBackend;

use BigBlueButton\Parameters\CreateMeetingParameters;
use BigBlueButton\Parameters\JoinMeetingParameters;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\key\KeyRepositoryInterface;
use Drupal\meeting_api\Attribute\Backend;
use Drupal\meeting_api\BackendPluginBase;
use Drupal\meeting_api\Exception\ConfigurationException;
use Drupal\meeting_api\Exception\OperationException;
use Drupal\meeting_api\MeetingAttendeeInterface;
use Drupal\meeting_api\MeetingInterface;
use Drupal\meeting_api_bbb\BigBlueButtonClient;
use Drupal\meeting_api_bbb\ClientFactory;
use Drupal\meeting_api_bbb\PluginForm\BigBlueButtonMeetingConfigurationForm;
use Drupal\meeting_api_bbb\PluginForm\BigBlueButtonServerConfigurationForm;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A plugin that allows to use BigBlueButton as backend.
 */
#[Backend(
  id: 'bigbluebutton',
  label: new TranslatableMarkup('BigBlueButton'),
  description: new TranslatableMarkup('BigBlueButton backend.'),
  forms: [
    'configure' => BigBlueButtonServerConfigurationForm::class,
    'meeting' => BigBlueButtonMeetingConfigurationForm::class,
  ]
)]
class BigBlueButton extends BackendPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The Big Blue Button client.
   *
   * @var \Drupal\meeting_api_bbb\BigBlueButtonClient|null
   */
  protected ?BigBlueButtonClient $client = NULL;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected ClientFactory $clientFactory,
    protected KeyRepositoryInterface $keyRepository,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(ClientFactory::class),
      $container->get('key.repository'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'url' => '',
      'key' => '',
    ];
  }

  /**
   * Retrieves BBB client.
   *
   * @return \Drupal\meeting_api_bbb\BigBlueButtonClient
   *   The BigBlueButton client.
   *
   * @throws \Drupal\meeting_api\Exception\ConfigurationException
   *   Thrown when the key is not found or secret is empty.
   */
  protected function getClient(): BigBlueButtonClient {
    if ($this->client === NULL) {
      $url = $this->configuration['url'];
      $key = $this->configuration['key'];
      $secret = $this->keyRepository->getKey($key)?->getKeyValue();

      if ($secret === NULL) {
        throw new ConfigurationException(sprintf('The configured key "%s" is not available.', $key));
      }

      $this->client = $this->clientFactory->createClient($url, $secret);
    }

    return $this->client;
  }

  /**
   * {@inheritdoc}
   */
  public function joinMeeting(MeetingInterface $meeting, MeetingAttendeeInterface $user): string {
    $meeting_id = $meeting->getIdentifier();
    // Create the meeting to join, the create call is idempotent.
    $create_meeting_params = new CreateMeetingParameters($meeting_id, $meeting->label() ?? $meeting_id);

    $settings = $meeting->getSettings();
    $create_meeting_params
      ->setRecord($settings['allow_recording'] ?? TRUE)
      ->setAutoStartRecording($settings['auto_start_recording'] ?? FALSE)
      ->setLockSettingsDisableCam(!($settings['allow_webcams'] ?? TRUE))
      ->setLockSettingsDisablePublicChat(!($settings['allow_public_chat'] ?? TRUE))
      ->setLockSettingsDisablePrivateChat(!($settings['allow_private_chat'] ?? FALSE))
      ->setWelcomeMessage($settings['welcome_message'] ?? '');

    // @todo Catch any BadResponseException|RuntimeException?
    $create_meeting = $this->getClient()->createMeeting($create_meeting_params);
    if (!$create_meeting->success()) {
      throw new OperationException(sprintf('There was a problem with the join request. Reason: "%s".', $create_meeting->getMessage()));
    }

    // Name is required for the join meeting operation, could be the case that
    // the display name returns an empty string or the name is altered by a hook
    // resulting into the same.
    $name = $user->getName();
    if ($name === '') {
      throw new OperationException('Missing user name.');
    }

    $user_role = $this->mapRole($user->getMeetingRole());
    $join_meeting_params = new JoinMeetingParameters($meeting_id, $name, $user_role);
    $join_meeting_params->setRedirect(TRUE);

    // The method joinMeeting() and XML response would bypass the redirection
    // and cookie needed for join operation hence we use getJoinMeetingURL() to
    // get the URL and allow the cookie creation.
    return $this->getClient()->getUrlBuilder()->getJoinMeetingURL($join_meeting_params);
  }

  /**
   * Maps meeting roles with BBB specific.
   *
   * @param string $role
   *   The role to map.
   *
   * @return string
   *   The resulting BBB role.
   */
  protected function mapRole(string $role): string {
    return match ($role) {
      MeetingAttendeeInterface::ATTENDEE_ROLE => 'VIEWER',
      MeetingAttendeeInterface::MODERATOR_ROLE => 'MODERATOR',
    };
  }

}
