<?php

declare(strict_types=1);

namespace Drupal\meeting_api_bbb;

use BigBlueButton\BigBlueButton;
use BigBlueButton\Parameters\DeleteRecordingsParameters;
use BigBlueButton\Parameters\GetMeetingInfoParameters;
use BigBlueButton\Parameters\GetRecordingTextTracksParameters;
use BigBlueButton\Parameters\InsertDocumentParameters;
use BigBlueButton\Parameters\IsMeetingRunningParameters;
use BigBlueButton\Parameters\PublishRecordingsParameters;
use BigBlueButton\Parameters\PutRecordingTextTrackParameters;
use BigBlueButton\Parameters\UpdateRecordingsParameters;
use BigBlueButton\Responses\DeleteRecordingsResponse;
use BigBlueButton\Responses\GetMeetingInfoResponse;
use BigBlueButton\Responses\GetMeetingsResponse;
use BigBlueButton\Responses\GetRecordingsResponse;
use BigBlueButton\Responses\GetRecordingTextTracksResponse;
use BigBlueButton\Responses\HooksCreateResponse;
use BigBlueButton\Responses\HooksDestroyResponse;
use BigBlueButton\Responses\HooksListResponse;
use BigBlueButton\Responses\IsMeetingRunningResponse;
use BigBlueButton\Responses\PublishRecordingsResponse;
use BigBlueButton\Responses\PutRecordingTextTrackResponse;
use BigBlueButton\Responses\UpdateRecordingsResponse;
use BigBlueButton\Parameters\CreateMeetingParameters;
use BigBlueButton\Parameters\EndMeetingParameters;
use BigBlueButton\Parameters\JoinMeetingParameters;
use BigBlueButton\Responses\CreateMeetingResponse;
use BigBlueButton\Responses\EndMeetingResponse;
use BigBlueButton\Responses\JoinMeetingResponse;
use Drupal\meeting_api\Exception\OperationException;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * {@inheritDoc}
 *
 * @internal
 */
final class BigBlueButtonClient extends BigBlueButton {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    string $baseUrl,
    #[\SensitiveParameter]
    string $secret,
    protected ClientInterface $httpClient,
  ) {
    parent::__construct($baseUrl, $secret);
  }

  /**
   * {@inheritDoc}
   */
  public function createMeeting(CreateMeetingParameters $createMeetingParams): CreateMeetingResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getCreateMeetingUrl($createMeetingParams), $createMeetingParams->getPresentationsAsXML());

    return new CreateMeetingResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function joinMeeting(JoinMeetingParameters $joinMeetingParams): JoinMeetingResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getJoinMeetingURL($joinMeetingParams));

    return new JoinMeetingResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function endMeeting(EndMeetingParameters $endParams): EndMeetingResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getEndMeetingURL($endParams));

    return new EndMeetingResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function insertDocument(InsertDocumentParameters $insertDocumentParams): CreateMeetingResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getInsertDocumentUrl($insertDocumentParams), $insertDocumentParams->getPresentationsAsXML());

    return new CreateMeetingResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function isMeetingRunning(IsMeetingRunningParameters $meetingParams): IsMeetingRunningResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getIsMeetingRunningUrl($meetingParams));

    return new IsMeetingRunningResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function getMeetings(): GetMeetingsResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getMeetingsUrl());

    return new GetMeetingsResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function getMeetingInfo(GetMeetingInfoParameters $meetingParams): GetMeetingInfoResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getMeetingInfoUrl($meetingParams));

    return new GetMeetingInfoResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function getRecordings($recordingParams): GetRecordingsResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getRecordingsUrl($recordingParams));

    return new GetRecordingsResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function publishRecordings(PublishRecordingsParameters $recordingParams): PublishRecordingsResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getPublishRecordingsUrl($recordingParams));

    return new PublishRecordingsResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function deleteRecordings(DeleteRecordingsParameters $recordingParams): DeleteRecordingsResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getDeleteRecordingsUrl($recordingParams));

    return new DeleteRecordingsResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function updateRecordings(UpdateRecordingsParameters $recordingParams): UpdateRecordingsResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getUpdateRecordingsUrl($recordingParams));

    return new UpdateRecordingsResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function getRecordingTextTracks(GetRecordingTextTracksParameters $getRecordingTextTracksParams): GetRecordingTextTracksResponse {
    $json = $this->processJsonResponse($this->getUrlBuilder()->getRecordingTextTracksUrl($getRecordingTextTracksParams));

    return new GetRecordingTextTracksResponse($json);
  }

  /**
   * {@inheritDoc}
   */
  public function putRecordingTextTrack(PutRecordingTextTrackParameters $putRecordingTextTrackParams): PutRecordingTextTrackResponse {
    $json = $this->processJsonResponse($this->getUrlBuilder()->getPutRecordingTextTrackUrl($putRecordingTextTrackParams));

    return new PutRecordingTextTrackResponse($json);
  }

  /**
   * {@inheritDoc}
   */
  public function hooksCreate($hookCreateParams): HooksCreateResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getHooksCreateUrl($hookCreateParams));

    return new HooksCreateResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function hooksList(): HooksListResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getHooksListUrl());

    return new HooksListResponse($xml);
  }

  /**
   * {@inheritDoc}
   */
  public function hooksDestroy($hooksDestroyParams): HooksDestroyResponse {
    $xml = $this->processXmlResponse($this->getUrlBuilder()->getHooksDestroyUrl($hooksDestroyParams));

    return new HooksDestroyResponse($xml);
  }

  /**
   * Sends requests using http_client service.
   *
   * @param string $url
   *   The URL to send the request.
   * @param string $payload
   *   The content of the body.
   * @param string $contentType
   *   The type of content send in the request.
   *
   * @throws \Drupal\meeting_api\Exception\OperationException
   *
   * @return string
   *   The resulting XML body content.
   */
  private function sendRequest(string $url, string $payload = '', string $contentType = 'xml'): string {
    $method = 'GET';
    $options = [
      'headers' => [
        'Content-type' => $contentType,
      ],
    ];

    if (!empty($payload)) {
      $method = 'POST';
      $options['body'] = $payload;
      $options['headers']['Content-length'] = mb_strlen($payload);
    }

    try {
      $response = $this->httpClient->request($method, $url, $options);
    }
    catch (GuzzleException $e) {
      throw new OperationException('There was a problem with the request.', previous: $e);
    }

    return $response->getBody()->getContents();
  }

  /**
   * Process XML responses.
   *
   * @param string $url
   *   The URL to send the request.
   * @param string $payload
   *   The content of the body.
   *
   * @return \SimpleXMLElement
   *   The resulting XMl body content.
   */
  protected function processXmlResponse(string $url, string $payload = ''): \SimpleXMLElement {
    $response = $this->sendRequest($url, $payload, 'xml');

    return new \SimpleXMLElement($response);
  }

  /**
   * Process json responses.
   *
   * @param string $url
   *   The URL to send the request.
   * @param string $payload
   *   The content of the body.
   *
   * @return string
   *   The resulting JSON body content.
   */
  protected function processJsonResponse(string $url, string $payload = ''): string {
    return $this->sendRequest($url, $payload, 'json');
  }

}
