<?php

namespace Drupal\Tests\meeting_api_bbb\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\meeting_api\Entity\MeetingType;
use Drupal\meeting_api\Entity\Server;
use Drupal\meeting_api\MeetingEntityInterface;
use Drupal\Tests\meeting_api\Traits\MeetingFormTrait;

/**
 * Tests the meeting settings.
 *
 * @group meeting_api_bbb
 */
class MeetingSettingsTest extends WebDriverTestBase {

  use MeetingFormTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'meeting_api_bbb',
  ];

  /**
   * Tests the settings available in the meeting entity form.
   */
  public function testMeetingFormSettings(): void {
    Server::create([
      'id' => 'bbb_server',
      'label' => 'BigBlueButton server',
      'backend' => 'bigbluebutton',
    ])->save();

    MeetingType::create([
      'id' => 'default',
      'label' => 'Default meeting type',
      'server_id' => 'bbb_server',
    ])->save();

    $this->drupalLogin($this->drupalCreateUser([
      'administer meeting_api_meeting entities',
    ]));

    $this->drupalGet('/meeting/add/default');
    $label = $this->fillMeetingMandatoryFields();
    $assert_session = $this->assertSession();
    $record_checkbox = $assert_session->fieldExists('Allow the meeting to be recorded');
    $this->assertTrue($record_checkbox->isChecked());
    $autostart_checkbox = $assert_session->fieldExists('Auto start recording');
    $this->assertFalse($autostart_checkbox->isChecked());
    $webcams_checkbox = $assert_session->fieldExists('Allow webcams');
    $this->assertTrue($webcams_checkbox->isChecked());
    $public_chat_checkbox = $assert_session->fieldExists('Allow public chat');
    $this->assertTrue($public_chat_checkbox->isChecked());
    $private_chat_checkbox = $assert_session->fieldExists('Allow private chat');
    $this->assertFalse($private_chat_checkbox->isChecked());
    $welcome_text_field = $assert_session->fieldExists('Welcome text');
    $this->assertEmpty($welcome_text_field->getValue());

    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains("New meeting {$label} has been created.", 'status');

    $meeting_storage = \Drupal::entityTypeManager()->getStorage('meeting_api_meeting');
    $meeting = array_values($meeting_storage->loadByProperties(['label' => $label]))[0];
    $this->assertInstanceOf(MeetingEntityInterface::class, $meeting);
    $this->assertEquals([
      'allow_recording' => TRUE,
      'auto_start_recording' => FALSE,
      'allow_webcams' => TRUE,
      'allow_public_chat' => TRUE,
      'allow_private_chat' => FALSE,
      'welcome_text' => '',
    ], $meeting->getSettings());

    // Check that the values are populated back correctly.
    $this->drupalGet($meeting->toUrl('edit-form'));
    $this->assertTrue($record_checkbox->isChecked());
    $this->assertFalse($autostart_checkbox->isChecked());
    $this->assertTrue($webcams_checkbox->isChecked());
    $this->assertTrue($public_chat_checkbox->isChecked());
    $this->assertFalse($private_chat_checkbox->isChecked());
    $this->assertEmpty($welcome_text_field->getValue());

    // Check that the autostart checkbox requires the record checkbox to be
    // active.
    $record_checkbox->uncheck();
    $assert_session->fieldDisabled('Auto start recording');

    // Check that the autostart checkbox will be ignored even if it's checked,
    // before getting disabled.
    $record_checkbox->check();
    $autostart_checkbox->check();
    $record_checkbox->uncheck();
    $assert_session->fieldDisabled('Auto start recording');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains("The meeting {$label} has been updated.", 'status');

    $meeting = $this->reloadEntity($meeting);
    $this->assertEquals([
      'allow_recording' => FALSE,
      'auto_start_recording' => FALSE,
      'allow_webcams' => TRUE,
      'allow_public_chat' => TRUE,
      'allow_private_chat' => FALSE,
      'welcome_text' => '',
    ], $meeting->getSettings());

    $this->drupalGet($meeting->toUrl('edit-form'));
    $assert_session->fieldDisabled('Auto start recording');
    $this->assertFalse($autostart_checkbox->isChecked());

    // Test other options.
    $record_checkbox->check();
    $autostart_checkbox->check();
    $webcams_checkbox->uncheck();
    $public_chat_checkbox->uncheck();
    $private_chat_checkbox->check();
    $welcome_text = $this->getRandomGenerator()->sentences(12);
    $welcome_text_field->setValue($welcome_text);
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains("The meeting {$label} has been updated.", 'status');

    $meeting = $this->reloadEntity($meeting);
    $this->assertEquals([
      'allow_recording' => TRUE,
      'auto_start_recording' => TRUE,
      'allow_webcams' => FALSE,
      'allow_public_chat' => FALSE,
      'allow_private_chat' => TRUE,
      'welcome_text' => $welcome_text,
    ], $meeting->getSettings());
  }

  /**
   * Reloads a meeting entity, bypassing caches.
   *
   * @param \Drupal\meeting_api\MeetingEntityInterface $meeting
   *   The meeting entity to reload.
   *
   * @return \Drupal\meeting_api\MeetingEntityInterface
   *   The reloaded meeting entity.
   */
  protected function reloadEntity(MeetingEntityInterface $meeting): MeetingEntityInterface {
    $storage = \Drupal::entityTypeManager()->getStorage('meeting_api_meeting');
    $storage->resetCache([$meeting->id()]);
    $entity = $storage->load($meeting->id());
    assert($entity instanceof MeetingEntityInterface);

    return $entity;
  }

}
