<?php

declare(strict_types=1);

namespace Drupal\Tests\meeting_api_bbb\Kernel;

use Drupal\Core\Http\ClientFactory;
use Drupal\KernelTests\KernelTestBase;
use Drupal\key\Entity\Key;
use Drupal\key\KeyInterface;
use Drupal\meeting_api\Entity\Meeting;
use Drupal\meeting_api\Entity\MeetingType;
use Drupal\meeting_api\Entity\Server;
use Drupal\meeting_api\MeetingAttendeeInterface;
use Drupal\meeting_api\MeetingEntityInterface;
use Drupal\meeting_api\MeetingManagerInterface;
use Drupal\meeting_api\MeetingAttendee;
use Drupal\meeting_api\ServerInterface;
use Drupal\Tests\user\Traits\UserCreationTrait;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Response;
use Psr\Http\Message\ResponseInterface;

/**
 * Tests join meeting operation.
 *
 * @group meeting_api_bbb
 */
class BigBlueButtonTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'meeting_api',
    'meeting_api_bbb',
    'key',
    'user',
    'datetime_range_timezone',
  ];

  /**
   * Key used on API requests.
   */
  protected KeyInterface $key;

  /**
   * Server entity where operations are performed.
   */
  protected ServerInterface $server;

  /**
   * Meeting type entity used for the meeting.
   */
  protected MeetingType $meetingType;

  /**
   * Meeting used for the operations.
   */
  protected MeetingEntityInterface $meeting;

  /**
   * The URI of the last request made to the mock client.
   */
  protected ?string $mockClientRequestUri = NULL;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('meeting_api_server');
    $this->installEntitySchema('meeting_api_meeting_type');
    $this->installEntitySchema('meeting_api_meeting');
    $this->installEntitySchema('user');

    $key = Key::create(['id' => 'bbb_test_key_id']);
    $key->setKeyValue('1234');
    $key->save();

    $this->server = Server::create([
      'id' => 'server',
      'backend' => 'bigbluebutton',
      'label' => 'Server',
      'backend_config' => [
        'url' => 'https://bigbluebuttonserver.com',
        'key' => 'bbb_test_key_id',
      ],
    ]);
    $this->server->save();

    $this->meetingType = MeetingType::create([
      'id' => 'meeting_type',
      'server_id' => 'server',
      'label' => 'Meeting type',
    ]);
    $this->meetingType->save();

    $this->meeting = Meeting::create([
      'bundle' => 'meeting_type',
      'label' => 'Fancy meeting',
      // Use a fixed uuid to have a stable checksum.
      'uuid' => '4321',
    ]);
    $this->meeting->save();
  }

  /**
   * Tests successful join meeting operation.
   */
  public function testJoinMeeting(): void {
    $user_bob = $this->createUser(name: 'Bob');
    $meeting_manager = \Drupal::service(MeetingManagerInterface::class);
    $this->mockHttpClientFactory($this->createTestResponseFromFile('createMeeting'));

    // Join a meeting with minimum parameters.
    $this->assertSame(
      'https://bigbluebuttonserver.com/api/join?' .
        implode('&', [
          'meetingID=4321',
          'fullName=Bob',
          'role=VIEWER',
          'redirect=true',
          'checksum=330e84e2362846578a8d3247349bb785805481dfc8ab4c102261cc834c0c0652',
        ]),
      $meeting_manager->joinMeeting(
        $this->meeting,
        new MeetingAttendee($user_bob, MeetingAttendeeInterface::ATTENDEE_ROLE)
      ),
    );
    $this->assertEquals(
      'https://bigbluebuttonserver.com/api/create?name=Fancy+meeting&meetingID=4321&record=true&autoStartRecording=false&lockSettingsDisableCam=false&lockSettingsDisablePrivateChat=true&lockSettingsDisablePublicChat=false&checksum=c634ad319d041f189a0180a967662606e9abbc7f21cbfc70ba31c9ca6411dee9',
      $this->mockClientRequestUri,
    );

    // Set a different role for the user.
    $this->mockHttpClientFactory($this->createTestResponseFromFile('createMeeting'));
    $this->assertSame(
      'https://bigbluebuttonserver.com/api/join?' .
        implode('&', [
          'meetingID=4321',
          'fullName=Jay',
          'role=MODERATOR',
          'redirect=true',
          'checksum=adca11e4d6fb2ef0acd0bd79791286c15d614af852b3ed7946a9d9239e1a0da4',
        ]),
      $meeting_manager->joinMeeting(
        $this->meeting,
        new MeetingAttendee($this->createUser(name: 'Jay'), MeetingAttendeeInterface::MODERATOR_ROLE)
      ),
    );
    $this->assertEquals(
      'https://bigbluebuttonserver.com/api/create?name=Fancy+meeting&meetingID=4321&record=true&autoStartRecording=false&lockSettingsDisableCam=false&lockSettingsDisablePrivateChat=true&lockSettingsDisablePublicChat=false&checksum=c634ad319d041f189a0180a967662606e9abbc7f21cbfc70ba31c9ca6411dee9',
      $this->mockClientRequestUri,
    );

    // Set a different name for the meeting.
    $this->meeting->set('label', 'Fancy meeting 2')->save();
    $this->mockHttpClientFactory($this->createTestResponseFromFile('createMeeting'));
    $this->assertSame(
      'https://bigbluebuttonserver.com/api/join?' .
        implode('&', [
          'meetingID=4321',
          'fullName=Bob',
          'role=VIEWER',
          'redirect=true',
          'checksum=330e84e2362846578a8d3247349bb785805481dfc8ab4c102261cc834c0c0652',
        ]),
      $meeting_manager->joinMeeting(
        $this->meeting,
        new MeetingAttendee($user_bob, MeetingAttendeeInterface::ATTENDEE_ROLE)
      ),
    );
    $this->assertEquals(
      'https://bigbluebuttonserver.com/api/create?name=Fancy+meeting+2&meetingID=4321&record=true&autoStartRecording=false&lockSettingsDisableCam=false&lockSettingsDisablePrivateChat=true&lockSettingsDisablePublicChat=false&checksum=41e7ab55425a93c9b61589cf9bba18c26cb0c41dfa67319123a6ad76401fbdf0',
      $this->mockClientRequestUri,
    );

    $this->meeting->set('backend_settings', [
      'allow_recording' => FALSE,
      // Setting autostart to true when record is false has no sense logically
      // and it's prevented in the UI. Here is done for the sake of testing all
      // the options.
      'auto_start_recording' => TRUE,
      'allow_webcams' => FALSE,
      'allow_public_chat' => FALSE,
      'allow_private_chat' => TRUE,
      'welcome_message' => 'Example welcome message',
    ])->save();
    $this->mockHttpClientFactory($this->createTestResponseFromFile('createMeeting'));
    $this->assertSame(
      'https://bigbluebuttonserver.com/api/join?' .
      implode('&', [
        'meetingID=4321',
        'fullName=Bob',
        'role=VIEWER',
        'redirect=true',
        'checksum=330e84e2362846578a8d3247349bb785805481dfc8ab4c102261cc834c0c0652',
      ]),
      $meeting_manager->joinMeeting(
        $this->meeting,
        new MeetingAttendee($user_bob, MeetingAttendeeInterface::ATTENDEE_ROLE)
      ),
    );
    $this->assertEquals(
      'https://bigbluebuttonserver.com/api/create?name=Fancy+meeting+2&meetingID=4321&record=false&autoStartRecording=true&welcome=Example+welcome+message&lockSettingsDisableCam=true&lockSettingsDisablePrivateChat=false&lockSettingsDisablePublicChat=true&checksum=be145a0196762b092a11a2253f28e4ea18a4eb9f033a451ce537403404d20801',
      $this->mockClientRequestUri,
    );
  }

  /**
   * Mocks the HTTP client factory and related services to return a response.
   *
   * @param \Psr\Http\Message\ResponseInterface $response
   *   The response that should be returned.
   */
  protected function mockHttpClientFactory(ResponseInterface $response): void {
    $this->mockClientRequestUri = NULL;

    // Only the "request" method should be called once, using GET as method.
    $client = $this->getMockBuilder(Client::class)
      ->disableOriginalConstructor()
      ->disableOriginalClone()
      ->onlyMethods(['request'])
      ->getMock();
    $client
      ->expects($this->once())
      ->method('request')
      ->with(
        $this->equalTo('GET'),
        $this->anything(),
        $this->anything(),
      )
      ->willReturnCallback(function (string $method, $uri, array $options = []) use ($response) {
        $this->mockClientRequestUri = $uri;

        return $response;
      });

    $client_factory = $this->createMock(ClientFactory::class);
    $client_factory
      ->expects($this->once())
      ->method('fromOptions')
      ->willReturn($client);

    $this->container->get('kernel')->rebuildContainer();
    $this->container = $this->container->get('kernel')->getContainer();
    \Drupal::setContainer($this->container);

    $this->container->set(ClientFactory::class, $client_factory);
    $this->container->set('http_client_factory', $client_factory);
  }

  /**
   * Creates a response object given a fixtures file.
   *
   * @param string $filename
   *   The filename, without extension, contained in the fixtures folder.
   * @param int $status_code
   *   The response code. Defaults to 200.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   The response object.
   */
  protected function createTestResponseFromFile(string $filename, int $status_code = 200): ResponseInterface {
    $file_path = dirname(__DIR__, 2) . '/fixtures/' . $filename . '.xml';

    if (!file_exists($file_path)) {
      throw new \Exception('File not found: ' . $file_path);
    }

    return new Response($status_code, body: file_get_contents($file_path));
  }

}
