<?php

namespace Drupal\meeting_api_bbb\PluginForm;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\meeting_api\MeetingInterface;

/**
 * Configuration form for the "meeting" action of the BBB backend.
 */
class BigBlueButtonMeetingConfigurationForm extends PluginFormBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $entity = $form_state->get('meeting_entity');
    assert($entity instanceof MeetingInterface);
    $configuration = $entity->getSettings();

    $form['notes'] = [
      '#markup' => $this->t('<small><em><strong>Note:</strong> the following settings cannot be changed after the meeting has started.</small></em>'),
    ];

    // @todo Lock the settings once the meeting is started?
    $form['allow_recording'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow the meeting to be recorded'),
      '#description' => $this->t('Whether the meeting can be recorded or not.'),
      '#default_value' => $configuration['allow_recording'] ?? TRUE,
    ];

    $form['auto_start_recording'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto start recording'),
      '#description' => $this->t('Whether to automatically start recording when first participant joins.'),
      '#default_value' => $configuration['auto_start_recording'] ?? FALSE,
      '#states' => [
        'enabled' => [
          ':input[name="backend_settings[allow_recording]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['allow_webcams'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow webcams'),
      '#description' => $this->t('Whether to allow webcams to be shared during the meeting.'),
      '#default_value' => $configuration['allow_webcams'] ?? TRUE,
    ];

    $form['allow_public_chat'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow public chat'),
      '#description' => $this->t('Whether the public chat will be available during the meeting.'),
      '#default_value' => $configuration['allow_public_chat'] ?? TRUE,
    ];

    $form['allow_private_chat'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow private chat'),
      '#description' => $this->t('Whether the private, one-on-one participant chat will be available during the meeting.'),
      '#default_value' => $configuration['allow_private_chat'] ?? FALSE,
    ];

    $form['welcome_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Welcome text'),
      '#description' => $this->t('A welcome message that gets displayed on the chat window when the participant joins.'),
      '#default_value' => $configuration['welcome_text'] ?? '',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    // Convert boolean values from integer.
    // @todo Can we use a configuration schema model here?
    $allow_recording = (bool) $form_state->getValue('allow_recording');
    $form_state->setValue('allow_recording', $allow_recording);
    $auto_start_recording = (bool) $form_state->getValue('auto_start_recording');
    // When allow_recording is set to false, force auto-start to false too.
    $form_state->setValue('auto_start_recording', $allow_recording && $auto_start_recording);

    $values = ['allow_webcams', 'allow_public_chat', 'allow_private_chat'];
    foreach ($values as $value) {
      $form_state->setValue($value, (bool) $form_state->getValue($value));
    }
  }

}
