<?php

declare(strict_types=1);

namespace Drupal\meeting_api_bbb_test\Plugin\ServiceMock;

use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\http_request_mock\Attribute\ServiceMock;
use Drupal\http_request_mock\ServiceMockPluginInterface;
use GuzzleHttp\Psr7\Response;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Intercepts any HTTP request made to bigbluebuttonserver.com.
 */
#[ServiceMock(
  id: 'bigbluebutton_server',
  label: new TranslatableMarkup('BigBlueButton Server'),
  weight: 0,
)]
class BigBlueButtonServerPlugin extends PluginBase implements ServiceMockPluginInterface {

  /**
   * {@inheritdoc}
   */
  public function applies(RequestInterface $request, array $options): bool {
    return $request->getUri()->getHost() === 'bigbluebuttonserver.com';
  }

  /**
   * {@inheritdoc}
   */
  public function getResponse(RequestInterface $request, array $options): ResponseInterface {
    $uri = $request->getUri()->getPath();
    $query = $request->getUri()->getQuery();
    $params = [];

    parse_str($query, $params);

    // Any method call using meeting with id "2" will inevitably fail
    // throwing a Guzzle exception.
    if ($params['meetingID'] === '2') {
      throw new RequestException('You can not use the meeting ID "2".', $request);
    }

    return match ($uri) {
      '/api/create' => $this->getCreateMeeting($params),
      default => throw new \Exception(sprintf('No matching endpoint for URI "%s"', $uri)),
    };
  }

  /**
   * Returns create meeting response and different fail responses.
   *
   * @param array $params
   *   The query parameters used in the request.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   The resulting response.
   */
  protected function getCreateMeeting(array $params): ResponseInterface {
    // Checksum is always required.
    if (in_array($params['checksum'],
      [
        // Allowed create meeting checksums.
        '70e24a51b850fd6776099b58da5604ab3d5026ca371cc60e9861f4425ffc59d7',
        '1b93eeba835fcf1dbf7569c6c0c2f70d68e019cb49ad857cd18987d2df61636b',
      ],
    )) {
      return $this->getResponseXml('createMeeting');
    }

    return $this->getResponseXml('checksumError');
  }

  /**
   * Provides the XML content in fixtures folder given a file name.
   *
   * @param string $filename
   *   The file name to look for.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   The content of the XML.
   */
  public function getResponseXml(string $filename): ResponseInterface {
    $file_path = dirname(__DIR__, 5) . '/fixtures/' . $filename . '.xml';

    if (!file_exists($file_path)) {
      throw new \Exception(sprintf('File "%s" not found.', $filename));
    }

    return new Response(200, ['content-type' => 'application/xml'], file_get_contents($file_path));
  }

}
