<?php

declare(strict_types=1);

namespace Drupal\Tests\meeting_api_bbb\Functional;

use Drupal\key\Entity\Key;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\meeting_api\Traits\AssertOptionsTrait;

/**
 * Tests backend management.
 *
 * @group meeting_api_bbb
 */
class BackendManagementTest extends BrowserTestBase {

  use AssertOptionsTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'meeting_api_bbb',
  ];

  /**
   * Tests the backend form.
   */
  public function testBackendForm(): void {
    $assert_session = $this->assertSession();
    $user = $this->createUser([
      'access administration pages',
      'administer meeting_api_server',
      'administer site configuration',
      'view the administration theme',
    ]);
    $this->drupalLogin($user);

    $bbb_key = Key::create([
      'id' => 'bbb_test_key_id',
      'label' => 'BigBlueButton test key',
      'key_type' => 'authentication',
      'key_provider' => 'config',
    ]);
    $bbb_key->save();

    Key::create([
      'id' => 'other_test_key_id',
      'label' => 'Other test key',
      'key_type' => 'encryption',
      'key_provider' => 'config',
    ])->save();

    // Select BBB as backend and try to save to trigger required fields errors.
    $this->drupalGet('admin/config/services/meeting-api-server/add');
    $assert_session->fieldExists('Label')->setValue('BigBlueButton Server');
    $assert_session->fieldExists('Machine-readable name')->setValue('bbb_server');
    $assert_session->fieldExists('Backend')->selectOption('BigBlueButton');
    $assert_session->buttonExists('Change backend')->press();
    $this->assertSelectOptions([
      '- Select a key -' => '- Select a key -',
      'bbb_test_key_id' => 'BigBlueButton test key',
    ], 'Key');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('Server URL field is required.', 'error');
    $assert_session->statusMessageContains('Key field is required.', 'error');

    // Set values for the backend.
    $assert_session->fieldExists('Server URL')->setValue('https://bigbluebuttonserver.com');
    $assert_session->fieldExists('Key')->selectOption('BigBlueButton test key');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('Created new server BigBlueButton Server.', 'status');

    // Values are updated.
    $this->drupalGet('admin/config/services/meeting-api-server/bbb_server');
    $assert_session->fieldValueEquals('Server URL', 'https://bigbluebuttonserver.com');
    $assert_session->fieldValueEquals('Key', 'bbb_test_key_id');

    // Set incorrect values to trigger field validation errors.
    $assert_session->fieldExists('Server URL')->setValue('bigbluebuttonserver.com');
    $assert_session->buttonExists('Save')->press();
    $assert_session->statusMessageContains('The URL bigbluebuttonserver.com is not valid.', 'error');
  }

}
