/**
 * @file
 * Mentions Tagify - Tagify autocomplete integration for Mentions module.
 *
 * Provides rich autocomplete dropdown with avatars when typing @mentions.
 * Converts Tagify's [[{json}]] format to Mentions module's @username format.
 * Supports multiple text formats with different Tagify configurations.
 */

(function ($, Drupal, once, Tagify) {
  'use strict';

  // Verify Tagify library is available.
  if (typeof Tagify === 'undefined') {
    console.error('Mentions Tagify: Tagify library not loaded!');
    return;
  }

  /**
   * Highlights matching text in autocomplete dropdown.
   *
   * @param {string} inputTerm - The full text to highlight within.
   * @param {string} searchTerm - The text to highlight.
   * @return {string} HTML string with matching text wrapped in <strong>.
   */
  function highlightMatchingLetters(inputTerm, searchTerm) {
    const escapedSearchTerm = searchTerm.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    const regex = new RegExp(`(${escapedSearchTerm})`, 'gi');
    if (!escapedSearchTerm) {
      return inputTerm;
    }
    return inputTerm.replace(regex, '<strong>$1</strong>');
  }

  /**
   * Template for inline mention tags.
   * Renders as: @username with remove button.
   *
   * @param {Object} tagData - Tag data from Tagify.
   * @return {string} HTML string for the tag.
   */
  function tagTemplate(tagData) {
    const label = tagData.label ?? tagData.value;
    return `<tag title="${label}"
      contenteditable='false'
      spellcheck='false'
      tabIndex="-1"
      class="tagify__tag ${tagData.class ? tagData.class : ''}"
      ${this.getAttributes(tagData)}>
        <x title='Remove ${label}'
          class='tagify__tag__removeBtn'
          role='button'
          aria-label='remove ${label} tag'>
        </x>
        <div>
          <span class='tagify__tag-text'>@${label}</span>
        </div>
    </tag>`;
  }

  /**
   * Template for autocomplete dropdown items.
   * Shows avatar and optional info label below username.
   *
   * @param {Object} tagData - Tag data from Tagify including avatar and info_label.
   * @return {string} HTML string for the dropdown item.
   */
  function suggestionItemTemplate(tagData) {
    // Guard against invalid data.
    if (!tagData || !tagData.label) {
      return '';
    }

    const searchTerm = tagData.input || this.state.inputText || '';

    return `<div ${this.getAttributes(tagData)}
      class='tagify__dropdown__item tagify__dropdown__item-center ${tagData.class ? tagData.class : ''}'
      tabindex="0"
      role="option">
      ${tagData.avatar
      ? `<div class='tagify__dropdown__item__avatar-wrap'><img onerror="this.style.visibility='hidden'" src="${tagData.avatar}"></div>`
      : ''
    }
      <div class="tagify__dropdown-user-info">
        <div class="tagify__dropdown-user-info-name">
          ${highlightMatchingLetters(tagData.label, searchTerm)}
        </div>
        ${tagData.info_label
      ? `<div class="tagify__dropdown-user-info-label"><small>${tagData.info_label}</small></div>`
      : ''
    }
      </div>
    </div>`;
  }

  /**
   * Handles autocomplete AJAX requests.
   * Fetches users from tagify_user_list controller and populates dropdown.
   *
   * @param {Tagify} tagify - Tagify instance.
   * @param {string} autocompleteUrl - URL to fetch autocomplete data from.
   * @param {string} storagePrefix - Mention prefix (e.g., '@').
   * @param {string} storageSuffix - Mention suffix (e.g., '').
   * @param {string} inputvalue - Field to use as value ('uid' or 'name').
   * @param {string} searchTerm - User's search input.
   */
  function handleAutocomplete(tagify, autocompleteUrl, storagePrefix, storageSuffix, inputvalue, searchTerm) {
    // Abort previous request if still pending.
    if (tagify._mentionsController) {
      tagify._mentionsController.abort();
    }

    tagify._mentionsController = new AbortController();

    if (searchTerm !== '') {
      tagify.loading(true);
    }

    // Build URL with search term and already-selected users.
    const url = new URL(autocompleteUrl, window.location.origin);
    url.searchParams.set('q', searchTerm);
    url.searchParams.set('selected', tagify.value
      .filter(item => item.entity_id)
      .map(item => item.entity_id)
      .join(','),
    );

    fetch(url.toString(), {
      signal: tagify._mentionsController.signal,
    })
      .then((res) => res.json())
      .then(function (newWhitelist) {
        // Transform backend data to Tagify format.
        const newWhitelistData = newWhitelist.map((current) => {
          let value;

          // Determine value based on configured inputvalue type.
          if (inputvalue === 'uid') {
            value = current.entity_id;
          } else if (inputvalue === 'name') {
            // Extract username from info_label if present.
            if (current.info_label && current.info_label.startsWith('@')) {
              value = current.info_label.substring(1);
            } else if (current.info_label) {
              value = current.info_label;
            } else {
              value = current.label;
            }
          } else {
            value = current.entity_id;
          }

          return {
            value: value,
            entity_id: current.entity_id,
            label: current.label,
            info_label: current.info_label,
            avatar: current.avatar,
            editable: current.editable,
            input: searchTerm,
          };
        });

        if (newWhitelistData.length > 0) {
          tagify.whitelist = newWhitelistData;
          tagify.loading(false).dropdown.show(searchTerm);
        } else {
          tagify.loading(false);
        }
      })
      .catch((error) => {
        // Ignore aborted requests (user typed more).
        if (!(error instanceof Error && error.name === 'AbortError')) {
          console.error('Mentions Tagify: Autocomplete failed:', error);
        }
        tagify.loading(false);
      });
  }

  /**
   * Initializes Tagify on a textarea with configuration for current format.
   *
   * Sets up:
   * - Mix mode (inline mentions with text)
   * - @ trigger for autocomplete
   * - Avatar display in dropdown
   * - Format conversion (Tagify JSON → Mentions format)
   *
   * @param {HTMLTextAreaElement} textarea - The textarea to initialize.
   * @param {Object} formatsConfig - Configuration for all available formats.
   * @param {string} currentFormat - The currently selected format ID.
   */
  async function initializeTagify(textarea, formatsConfig, currentFormat) {
    // Prevent double initialization.
    if (textarea.tagifyInstance || textarea.classList.contains('tagify__input')) {
      return;
    }

    textarea.setAttribute('data-mentions-tagify-active-text-format', currentFormat);

    // Get config for current format.
    const formatConfig = formatsConfig[currentFormat];

    if (!formatConfig || !formatConfig.enabled) {
      return;
    }

    // Currently only supports one set of mentions.
    const mentionType = formatConfig.mentions[0];

    if (!mentionType || !mentionType.autocomplete_url) {
      return;
    }

    // Always use @ as trigger for autocomplete.
    const TRIGGER = '@';

    // Storage format from MentionsType config (usually @username).
    const storagePrefix = mentionType.prefix || '@';
    const storageSuffix = mentionType.suffix || '';

    const inputvalue = mentionType.inputvalue || 'name';
    const autocompleteUrl = mentionType.autocomplete_url;
    const maxItems = formatConfig.max_items || 10;
    const minHeight = formatConfig.min_height || 100;
    const maxHeight = formatConfig.max_height || 300;

    // Convert existing @username mentions to Tagify format for editing.
    await parseExistingMentions(textarea, storagePrefix, storageSuffix, inputvalue);

    try {
      const tagify = new Tagify(textarea, {
        mode: 'mix',
        pattern: TRIGGER,
        mixTagsInterpolator: ['[[', ']]'],
        dropdown: {
          enabled: 0,
          highlightFirst: true,
          maxItems: maxItems,
          closeOnSelect: true,
          position: 'text',
          searchKeys: ['label', 'info_label'],
          mapValueTo: 'label',
        },
        templates: {
          tag: tagTemplate,
          dropdownItem: suggestionItemTemplate,
        },
        whitelist: [],
        tagTextProp: 'label',
        editTags: false,
        duplicates: false,
        enforceWhitelist: false,
      });

      // Set textarea height constraints.
      const tagifyWrapper = tagify.DOM.scope;
      tagifyWrapper.style.minHeight = minHeight + 'px';

      if (maxHeight > 0) {
        tagifyWrapper.style.maxHeight = maxHeight + 'px';
        tagifyWrapper.style.overflowY = 'auto';
      }

      // Debounced autocomplete on user input.
      const onInput = Drupal.debounce(function (e) {
        if (e.detail.prefix !== TRIGGER) {
          return;
        }

        const searchTerm = e.detail.value || '';
        handleAutocomplete(tagify, autocompleteUrl, storagePrefix, storageSuffix, inputvalue, searchTerm);
      }, 250);

      tagify.on('input', onInput);

      // Clear whitelist after selection to force fresh results.
      tagify.on('add', function () {
        tagify.whitelist = [];
      });

      // Store instance for cleanup.
      textarea.tagifyInstance = tagify;

    } catch (error) {
      console.error('Mentions Tagify: Initialization failed:', error);
    }
  }

  /**
   * Destroys Tagify instance and cleans up.
   *
   * @param {HTMLTextAreaElement} textarea - The textarea with Tagify instance.
   */
  function destroyTagify(textarea) {
    if (textarea && textarea.tagifyInstance) {
      try {
        textarea.tagifyInstance.destroy();
        textarea.tagifyInstance = null;
      } catch (error) {
        console.error('Mentions Tagify: Destroy failed:', error);
      }
    }
  }

  Drupal.behaviors.mentionsTagify = {
    attach: function (context, settings) {
      // Early return if no settings
      if (!settings.mentionsTagify) {
        return;
      }

      const formatsConfig = settings.mentionsTagify.formats || {};

      // Find format selectors with our attribute
      const selectors = once('mentions-tagify', '[data-mentions-tagify-for]', context);

      selectors.forEach(function (selector) {
        const fieldId = selector.getAttribute('data-mentions-tagify-for');
        const textarea = document.getElementById(fieldId);

        if (!textarea) {
          return;
        }

        // Get current format from selector value
        const activeFormatID = selector.value;
        textarea.setAttribute('data-mentions-tagify-active-text-format', activeFormatID);

        // Initialize if it is a tagify format.
        if (formatsConfig[activeFormatID]) {
          initializeTagify(textarea, formatsConfig, activeFormatID);
        }

        // Attach change listener
        if (selector.tagName === 'SELECT') {
          selector.addEventListener('change', function () {
            const newFormatID = this.value;
            const currentFormatID = textarea.getAttribute('data-mentions-tagify-active-text-format');

            // Prevent double-initialization
            if (newFormatID === currentFormatID) {
              return;
            }

            // Destroy old instance if exists
            if (textarea.tagifyInstance) {
              textarea.tagifyInstance.destroy();
              textarea.tagifyInstance = null;
            }

            // Update the attribute
            textarea.setAttribute('data-mentions-tagify-active-text-format', newFormatID);

            // Initialize if new format has Tagify
            if (formatsConfig[newFormatID]) {
              initializeTagify(textarea, formatsConfig, newFormatID);
            }
          });
        }
      });
    },

    detach: function (context, settings, trigger) {
      if (trigger === 'unload') {
        // ✅ Remove our once() binding
        const selectors = once.remove('mentions-tagify', '[data-mentions-tagify-for]', context);

        // Destroy Tagify instances
        selectors.forEach(function (selector) {
          const fieldId = selector.getAttribute('data-mentions-tagify-for');
          const textarea = document.getElementById(fieldId);
          if (textarea) {
            destroyTagify(textarea);
          }
        });
      }
    },
  };

  /**
   * Parses existing mentions in textarea and converts to Tagify format.
   *
   * When editing content, converts:
   *   @username → [[{"value":"username","label":"Full Name"}]]
   *
   * This allows Tagify to recognize existing mentions and make them editable.
   * Securely loads user data via POST with CSRF protection.
   *
   * @param {HTMLTextAreaElement} textarea - The textarea element.
   * @param {string} storagePrefix - Mention prefix to match (e.g., '@').
   * @param {string} storageSuffix - Mention suffix to match (e.g., '').
   * @param {string} inputvalue - Field type ('uid' or 'name').
   */
  async function parseExistingMentions(textarea, storagePrefix, storageSuffix, inputvalue) {
    const content = textarea.value;

    // Build regex pattern from configured storage format.
    const escapedPrefix = storagePrefix.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    const escapedSuffix = storageSuffix.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    const pattern = new RegExp(escapedPrefix + '([a-zA-Z0-9_]+)' + escapedSuffix, 'g');

    const matches = [...content.matchAll(pattern)];

    if (matches.length === 0) {
      return;
    }

    // Extract unique usernames.
    const values = [...new Set(matches.map(m => m[1]))];

    try {
      const token = await getCsrfToken();

      // Load user data from backend.
      const response = await fetch('/mentions_tagify/load-mention-data', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-CSRF-Token': token,
        },
        body: JSON.stringify({
          target_type: 'user',
          inputvalue: inputvalue,
          values: values,
        }),
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const entities = await response.json();

      // Build lookup map: username → display name.
      const entityMap = {};
      entities.forEach(entity => {
        entityMap[entity.value] = entity.label;
      });

      // Replace each mention with Tagify format.
      let convertedContent = content;

      matches.forEach(match => {
        const matchedValue = match[1];
        const label = entityMap[matchedValue];

        if (label) {
          const tagifyJson = JSON.stringify({
            value: matchedValue,
            label: label,
          });

          convertedContent = convertedContent.replace(
            match[0],
            `[[${tagifyJson}]]`,
          );
        }
        // If no label found, leave as plain text (user doesn't exist or no access).
      });

      textarea.value = convertedContent;

    } catch (error) {
      console.error('Mentions Tagify: Failed to parse existing mentions:', error);
      // Don't block initialization - continue with unparsed content.
    }
  }

  // ========================================================================
  // CSRF TOKEN MANAGEMENT
  // ========================================================================

  let csrfToken = null;

  /**
   * Gets CSRF token for secure POST requests.
   * Cached after first fetch.
   *
   * @return {Promise<string>} CSRF token.
   */
  async function getCsrfToken() {
    if (csrfToken) {
      return csrfToken;
    }

    try {
      const response = await fetch('/session/token');
      csrfToken = await response.text();
      return csrfToken;
    } catch (error) {
      console.error('Mentions Tagify: Failed to get CSRF token:', error);
      throw error;
    }
  }

})(jQuery, Drupal, once, window.Tagify);
