<?php

namespace Drupal\mentions_tagify\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

/**
 * Controller for loading mention data.
 */
class MentionsTagifyLoadController extends ControllerBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static();
  }

  /**
   * Loads mention data for existing mentions in text.
   *
   * Security measures:
   * - POST only with CSRF token
   * - Entity access check: Only returns users current user can view
   * - Input validation and sanitization
   * - Rate limiting: Max 50 values per request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with user data.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\BadRequestHttpException
   *   Thrown for invalid requests.
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   Thrown for unauthorized access.
   */
  public function loadMentions(Request $request): JsonResponse {
    // Parse JSON body.
    $data = json_decode($request->getContent(), TRUE);

    if (!is_array($data)) {
      throw new BadRequestHttpException('Invalid JSON payload');
    }

    // Validate target_type.
    $target_type = $data['target_type'] ?? NULL;
    if ($target_type !== 'user') {
      throw new AccessDeniedHttpException('Only user entities are supported');
    }

    // Validate inputvalue.
    $inputvalue = $data['inputvalue'] ?? NULL;
    if (!in_array($inputvalue, ['uid', 'name'], TRUE)) {
      throw new AccessDeniedHttpException('Invalid inputvalue parameter');
    }

    // Get values array.
    $values = $data['values'] ?? [];
    if (!is_array($values)) {
      throw new BadRequestHttpException('Values must be an array');
    }

    // Sanitize values based on inputvalue type.
    $values = array_filter($values, function ($v) use ($inputvalue) {
      if ($inputvalue === 'uid') {
        // UIDs must be positive integers.
        return is_numeric($v) && $v > 0 && $v == (int) $v;
      }
      else {
        // Usernames: alphanumeric and underscore only.
        return is_string($v) && preg_match('/^[a-zA-Z0-9_]+$/', $v);
      }
    });

    // Limit to prevent abuse.
    if (count($values) > 50) {
      throw new BadRequestHttpException('Maximum 50 values allowed per request');
    }

    if (empty($values)) {
      return new JsonResponse([]);
    }

    // Load user entities (use inherited entityTypeManager() method).
    $storage = $this->entityTypeManager()->getStorage('user');
    $results = [];

    foreach ($values as $value) {
      $user = NULL;

      try {
        if ($inputvalue === 'uid') {
          // Load by UID.
          $user = $storage->load($value);
        }
        else {
          // Load by username.
          $query = $storage->getQuery()
            ->condition('name', $value)
            ->accessCheck(TRUE)
            ->range(0, 1)
            ->execute();

          $uid = reset($query);
          if ($uid) {
            $user = $storage->load($uid);
          }
        }

        // Check entity access - only return if current user can view.
        if ($user && $user->access('view')) {
          $results[] = [
            'value' => (string) $value,
            'label' => $user->getDisplayName(),
          ];
        }
        // Silently skip users that don't exist or aren't accessible.
      }
      catch (\Exception $e) {
        // Log the error but don't expose it to the user.
        $this->getLogger('mentions_tagify')->error(
          'Error loading user mention: @message',
          ['@message' => $e->getMessage()]
        );
        // Continue processing other values.
      }
    }

    return new JsonResponse($results);
  }

}
