<?php

namespace Drupal\mentions_tagify\Element;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Render\Element\RenderCallbackInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\Url;
use Drupal\filter\Entity\FilterFormat;

/**
 * Provides render callbacks for mentions_tagify elements.
 *
 * Pattern: \Drupal\editor\Element\EditorElement.
 *
 * We follow the same approach as CKEditor:
 * - Format configs stored in drupalSettings.mentionsTagify.formats
 * - Each textarea only stores which formats it allows (data-allowed-formats)
 */
class MentionsTagifyElement implements TrustedCallbackInterface, RenderCallbackInterface {

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['preRenderTextFormat'];
  }

  /**
   * Pre-render callback for text format elements.
   *
   * When there's a format selector, we need to attach configs for ALL formats,
   * then JS will activate the right one based on current selection.
   *
   * Pattern: editor_pre_render_text_format()
   *
   * @param array $element
   *   The text format element.
   *
   * @return array
   *   The modified element.
   */
  public static function preRenderTextFormat(array $element) {
    // Determine which formats are available for this field.
    $format_ids = [];

    // If there's a format selector, get all allowed formats.
    if (isset($element['format']['format']['#options'])) {
      $format_ids = array_keys($element['format']['format']['#options']);
    }
    // Otherwise, use the single format.
    elseif (isset($element['#format'])) {
      $format_ids = [$element['#format']];
    }

    if (empty($format_ids)) {
      return $element;
    }

    // Build configs for each format that has Tagify enabled.
    // We'll store these globally in drupalSettings.
    $formats_config = [];

    foreach ($format_ids as $format_id) {
      $format = FilterFormat::load($format_id);

      if (!$format || !$format->filters()->has('filter_mentions')) {
        continue;
      }

      $filter = $format->filters('filter_mentions');
      $filter_config = $filter->getConfiguration();
      $tagify_settings = $filter_config['settings']['tagify'] ?? [];

      if (empty($tagify_settings['enabled'])) {
        continue;
      }

      // Get which mention types are enabled in this filter.
      $enabled_mentions = array_filter($filter_config['settings']['mentions_filter'] ?? []);

      if (empty($enabled_mentions)) {
        continue;
      }

      // Load mention type entities.
      $mentions_types = \Drupal::entityTypeManager()
        ->getStorage('mentions_type')
        ->loadMultiple($enabled_mentions);

      // Build configuration for each mention type (only user types for now).
      $mentions_config = [];
      foreach ($mentions_types as $type_id => $mentions_type) {
        $input = $mentions_type->getInputSettings();

        // Only support user mentions for now.
        if ($input['entity_type'] !== 'user') {
          continue;
        }

        // Build selection settings.
        $selection_settings = static::buildSelectionSettings($tagify_settings);

        // Create hashed key for security.
        $target_type = 'user';
        $selection_handler = $selection_settings['handler'];
        unset($selection_settings['handler']);

        $data = serialize($selection_settings) . $target_type . $selection_handler;
        $selection_settings_key = Crypt::hmacBase64($data, Settings::getHashSalt());

        // Store in key/value.
        $key_value_storage = \Drupal::keyValue('entity_autocomplete');
        if (!$key_value_storage->has($selection_settings_key)) {
          $key_value_storage->set($selection_settings_key, $selection_settings);
        }

        // Build autocomplete URL.
        $autocomplete_url = Url::fromRoute('mentions_tagify.entity_autocomplete', [
          'target_type' => $target_type,
          'selection_handler' => $selection_handler,
          'selection_settings_key' => $selection_settings_key,
        ])->toString();

        $mentions_config[] = [
          'type_id' => $type_id,
          'prefix' => $input['prefix'] ?? '@',
          'suffix' => $input['suffix'] ?? '',
          'entity_type' => $input['entity_type'],
          'autocomplete_url' => $autocomplete_url,
          'inputvalue' => $input['inputvalue'] ?? 'name',
          'output_pattern' => $output['outputvalue'] ?? '[user:name]',
        ];
      }

      if (!empty($mentions_config)) {
        // Store this format's config.
        $formats_config[$format_id] = [
          'enabled' => TRUE,
          'mentions' => $mentions_config,
          'max_items' => $tagify_settings['source']['match_limit'] ?? 10,
          'min_height' => $tagify_settings['textarea']['min_height'] ?? 100,
          'max_height' => $tagify_settings['textarea']['max_height'] ?? 300,
        ];
      }
    }

    // If no formats have Tagify enabled, exit.
    if (empty($formats_config)) {
      return $element;
    }

    // Get the textarea field ID.
    $field_id = $element['value']['#id'];

    // Determine the active format.
    $active_format = $element['#format'] ?? $format_ids[0];

    // Set active format on textarea.
    $element['value']['#attributes']['data-mentions-tagify-active-text-format'] = $active_format;

    if (!$element['format']['format']['#access']) {
      // Single format - create new hidden input with our attribute.
      $element['format']['mentions_tagify_selector'] = [
        '#type' => 'hidden',
        '#value' => $active_format,
        '#attributes' => [
          'data-mentions-tagify-for' => $field_id,
        ],
      ];
    }
    else {
      // Multiple formats - add attribute to existing SELECT.
      $element['format']['format']['#attributes']['data-mentions-tagify-for'] = $field_id;
    }

    // Attach library and settings to ROOT element.
    $element['#attached']['library'][] = 'mentions_tagify/mentions';
    $element['#attached']['drupalSettings']['mentionsTagify']['formats'] = $formats_config;
    return $element;
  }

  /**
   * Builds selection settings for tagify_user_list matcher.
   *
   * @param array $settings
   *   Tagify settings from filter configuration.
   *
   * @return array
   *   Selection settings for the matcher.
   */
  protected static function buildSelectionSettings(array $settings) {
    // Extract nested arrays.
    $source = $settings['source'] ?? [];
    $display = $settings['display'] ?? [];

    $reference_method = $source['reference_method'] ?? 'default';

    $selection_settings = [
      'handler' => $reference_method,
    ];

    // Views method.
    if ($reference_method === 'views' && !empty($source['view'])) {
      $view_parts = explode(':', $source['view'], 2);
      if (count($view_parts) === 2) {
        [$view_name, $display_id] = $view_parts;
        $arguments_string = trim($source['view_arguments'] ?? '');
        $arguments = ($arguments_string === '') ? [] : array_map('trim', explode(',', $arguments_string));

        $selection_settings['view'] = [
          'view_name' => $view_name,
          'display_name' => $display_id,
          'arguments' => $arguments,
        ];
      }
      else {
        $selection_settings['handler'] = 'default';
      }
    }

    // Default method with role filtering.
    if ($selection_settings['handler'] === 'default') {
      $selection_settings['include_anonymous'] = FALSE;
      if (!empty($source['role_limit'])) {
        $selection_settings['filter']['role'] = array_filter($source['roles'] ?? []);
      }
    }

    // Match settings.
    $selection_settings['match_operator'] = $source['match_operator'] ?? 'CONTAINS';
    $selection_settings['match_limit'] = $source['match_limit'] ?? 10;

    // Info label.
    $selection_settings['info_label'] = !empty($display['show_info_label'])
      ? ($display['info_label'] ?? '@[user:name]')
      : FALSE;

    // Avatar settings.
    $selection_settings['image'] = $display['image'] ?? 'user_picture';
    $selection_settings['image_style'] = $display['image_style'] ?? 'thumbnail';

    return $selection_settings;
  }

}
