<?php

declare(strict_types=1);

namespace Drupal\menu_export_csv\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Menu\MenuLinkInterface;
use Drupal\Core\Menu\MenuLinkManagerInterface;
use Drupal\Core\Menu\MenuLinkTreeInterface;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\Core\Utility\LinkGeneratorInterface;
use Drupal\menu_link_content\MenuLinkContentStorageInterface;
use Drupal\system\MenuInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * Returns responses for Menu export CSV routes.
 */
class MenuExportCsvController extends ControllerBase {

  /**
   * Constructs a MenuExportCsvController object.
   */
  public function __construct(
    protected MenuLinkManagerInterface $menuLinkManager,
    protected MenuLinkTreeInterface $menuTree,
    protected LinkGeneratorInterface $linkGenerator,
    protected MenuLinkContentStorageInterface $menuLinkContentStorage,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.menu.link'),
      $container->get('menu.link_tree'),
      $container->get('link_generator'),
      $container->get('entity_type.manager')->getStorage('menu_link_content')
    );
  }

  /**
   * Builds the menu export CSV response.
   *
   * @see \Drupal\menu_ui\MenuForm::buildOverviewForm
   * @see \Drupal\menu_ui\MenuForm::buildOverviewTreeForm
   */
  public function download(MenuInterface $menu): StreamedResponse|array {
    $tree = $this->menuTree->load($menu->id(), new MenuTreeParameters());

    $this->preloadEntities($tree);

    $response = new StreamedResponse(function() use ($menu, $tree): void {
      $handle = fopen('php://output', 'r+');

      // Header.
      $header = [
        'hierarchy',
        'depth',
        'title',
        'description',
        'url',
        'enabled',
        'expanded',
        'entity_type',
        'entity_id',
        'entity_status',
        'menu_link_id',
        'menu_link_provider',
      ];

      // Allow header to be altered.
      $context = ['menu' => $menu];
      $this->moduleHandler()->alter('menu_export_csv_header', $header, $context);

      fputcsv($handle, $header);

      // Rows.
      $rows = $this->getRows($menu, $tree);
      foreach ($rows as $row) {
        fputcsv($handle, $row);
      }

      fclose($handle);
    });

    $response->headers->set('Content-Type', 'application/force-download');
    $response->headers->set('Content-Disposition', 'attachment; filename="menu-' . $menu->id() . '.csv"');
    return $response;
  }

  /**
   * Retrieves and formats rows of menu data from a given menu tree structure.
   *
   * @param \Drupal\Core\Menu\MenuInterface $menu
   *   The menu object that provides context for the tree.
   * @param array $tree
   *   The hierarchical menu link tree array to process.
   * @param array &$rows
   *   (optional) An array of rows to be populated recursively. Defaults to an empty array.
   *
   * @return array
   *   An array of formatted rows, each containing menu data such as title, URL,
   *   description, hierarchy, enabled/disabled status, and associated entity information.
   */
  protected function getRows(MenuInterface $menu, array $tree, array &$rows = []): array {
    foreach ($tree as $menu_element) {
      /** @var \Drupal\Core\Menu\MenuLinkInterface $link */
      $link = $menu_element->link;

      $row = [];

      $row['hierarchy'] = str_repeat(' -', $menu_element->depth) . ' ' . $link->getTitle();
      $row['depth'] = $menu_element->depth;
      $row['title'] = $link->getTitle();
      $row['description'] = $link->getDescription();
      $row['url'] = $link->getUrlObject()->setAbsolute()->toString();
      $row['enabled'] = $link->isEnabled() ? $this->t('Yes') : $this->t('No');
      $row['expanded'] = $link->isExpanded() ? $this->t('Yes') : $this->t('No');
      $row['entity_type'] = '';
      $row['entity_id'] = '';
      $row['entity_status'] = '';
      $row['menu_link_id'] = $link->getPluginId();
      $row['menu_link_provider'] = $link->getProvider();

      // Get the link's target entity.
      $entity = $this->getMenuLinkRouteEntity($link);
      if ($entity) {
        $row['entity_type'] = $entity->getEntityTypeId();
        $row['entity_id'] = $entity->id();
        if ($entity instanceof EntityPublishedInterface) {
          $row['entity_status'] = $entity->isPublished() ? $this->t('Published') : $this->t('Unpublished');
        }
      }

      // Allow row to be altered.
      $context = [
        'menu' => $menu,
        'tree' => $tree,
        'menu_element' => $menu_element,
        'link' => $menu_element->link,
        'entity' => $entity,
      ];
      $this->moduleHandler()->alter('menu_export_csv_row', $row, $context);

      $rows[] = $row;

      if ($menu_element->subtree) {
        $this->getRows($menu, $menu_element->subtree, $rows);
      }
    }
    return $rows;
  }

  /**
   * Retrieves the entity associated with a menu link's route.
   *
   * @param \Drupal\Core\Menu\MenuLinkInterface $link
   *   The menu link for which the associated entity is to be determined.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The entity associated with the menu link's route, or NULL if no entity
   *   is found or the route does not reference an entity.
   */
  protected function getMenuLinkRouteEntity(MenuLinkInterface $link): ?EntityInterface {
    if (!preg_match('#^entity\.(.+?)\.canonical$#', $link->getRouteName(), $matches)) {
      return NULL;
    }

    $entity_type_id = $matches[1];
    if (!$this->entityTypeManager()->hasDefinition($entity_type_id)) {
      return NULL;
    }

    $entity_id = $link->getRouteParameters()[$entity_type_id] ?? NULL;
    if (!$entity_id) {
      return NULL;
    }

    return $this->entityTypeManager()
      ->getStorage($entity_type_id)
      ->load($entity_id);
  }

  /**
   * Preloads entities based on the provided tree structure.
   *
   * Using load multiple will cache all entities using an optimized query.
   *
   * @param array $tree
   *   The tree structure containing entity type IDs and their respective IDs.
   */
  protected function preloadEntities(array $tree): void {
    $entity_ids = $this->getEntityIds($tree);
    foreach ($entity_ids as $entity_type_id => $ids) {
      $this->entityTypeManager->getStorage($entity_type_id)->loadMultiple($ids);
    }
  }

  /**
   * Extracts entity IDs from a menu tree structure.
   *
   * @param array $tree
   *   The menu tree to process. Each element in the tree should be structured
   *   with 'link' and optionally 'subtree' properties.
   * @param array &$entity_ids
   *   (optional) An array to collect entity IDs, grouped by entity type.
   *
   * @return array
   *   An associative array of entity IDs, grouped by entity type ID.
   *   Example: ['node' => 1, 'taxonomy_term' => 2].
   */
  protected function getEntityIds(array $tree, array &$entity_ids = []): array {
    foreach ($tree as $element) {
      /** @var \Drupal\Core\Menu\MenuLinkInterface $link */
      $link = $element->link;
      if (preg_match('#^entity\.(.+?)\.canonical$#', $link->getRouteName(), $matches)) {
        $entity_type_id = $matches[1];
        if ($this->entityTypeManager()->hasDefinition($entity_type_id)) {
          $entity_ids += [$entity_type_id => []];
          $entity_ids[$entity_type_id][] = $link->getRouteParameters()[$entity_type_id];
        }
      }
      if ($element->subtree) {
        $this->getEntityIds($element->subtree, $entity_ids);
      }
    }
    return $entity_ids;
  }

}
