<?php

declare(strict_types=1);

namespace Drupal\menu_export_csv\EventSubscriber;

use Drupal\Core\DependencyInjection\ServiceProviderBase;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ViewEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Alter the menu edit form to add a 'Download CSV' link.
 *
 * @see \Drupal\menu_ui\MenuForm
 */
class MenuExportCsvEventSubscriber extends ServiceProviderBase implements EventSubscriberInterface {
  use StringTranslationTrait;

  /**
   * Constructs a MenuExportCsvEventSubscriber object.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The current route match.
   */
  public function __construct(protected RouteMatchInterface $routeMatch) {}

  /**
   * Alters Schema.org mapping list builder and adds a 'Download CSV' link.
   *
   * @param \Symfony\Component\HttpKernel\Event\ViewEvent $event
   *   The event to process.
   */
  public function onView(ViewEvent $event): void {
    $route_name = $this->routeMatch->getRouteName();
    if ($route_name === 'entity.menu.edit_form') {
      $result = $event->getControllerResult();
      $url = Url::fromRoute('menu_export_csv.download', ['menu' => $this->routeMatch->getRawParameter('menu')]);
      $result['export'] = [
        '#type' => 'link',
        '#title' => $this->t('<u>⇩</u> Download CSV'),
        '#url' => $url,
        '#attributes' => ['class' => ['button', 'button--small', 'button--extrasmall']],
        '#prefix' => '<p>',
        '#suffix' => '</p>',
        '#weight' => 100,
      ];
      $event->setControllerResult($result);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Run before main_content_view_subscriber.
    $events[KernelEvents::VIEW][] = ['onView', 100];
    return $events;
  }

}
