<?php

namespace Drupal\menu_family\FamilyRelation;

use Drupal\node\Entity\Node;
use Drupal\menu_link_content\Plugin\Menu\MenuLinkContent;

/**
 * Abstract class to find related nodes.
 */
abstract class Related {

  /**
   * The node ID.
   *
   * @var int
   */
  protected int $currentNodeId;

  /**
   * The display mode.
   *
   * @var string
   */
  protected string $displayMode;

  /**
   * The menu name.
   *
   * @var string
   */
  protected string $menuName;

  /**
   * Related constructor.
   *
   * @param int $current_node_id
   *   The current node ID.
   * @param string $display_mode
   *   The display mode.
   * @param string $menu_name
   *   The menu name.
   */
  public function __construct(
    int $current_node_id,
    string $display_mode,
    string $menu_name,
  ) {
    $this->currentNodeId = $current_node_id;
    $this->displayMode = $display_mode;
    $this->menuName = $menu_name;
  }

  /**
   * Generate related nodes.
   *
   * @return array
   *   The menu link content data.
   */
  public function generateNodes(): array {
    $menu_child_results = $this->getRelateds();

    $nodes_child = [];
    foreach ($menu_child_results as $key => $item) {
      if ($item === NULL) {
        $nodes_child[] = [];
        continue;
      }

      $link = \Drupal::entityTypeManager()
        ->getStorage('menu_link_content')
        ->load($item)
        ->getUrlObject();

      if (!$link->isRouted()) {
        continue;
      }

      if ($link->getRouteName() !== 'entity.node.canonical') {
        continue;
      }

      $params = $link->getRouteParameters();
      $node_id = $params['node'];

      $nodes_child[] = $this->generateViewMode($node_id);
    }

    return $nodes_child;
  }

  /**
   * Find the UUID of the first menu link content of a node.
   *
   * @param int $node_id
   *   The node ID.
   *
   * @return \Drupal\menu_link_content\Entity\MenuLinkContent|null
   *   The first menu link content found.
   */
  protected function findNodeMenuLinkContent(int $node_id): ?MenuLinkContent {
    $entries = \Drupal::service('plugin.manager.menu.link')->loadLinksByRoute(
      'entity.node.canonical',
      ['node' => $node_id],
      $this->menuName
    );

    if (empty($entries)) {
      return NULL;
    }

    return reset($entries);
  }

  /**
   * Render view mode for node.
   *
   * @param int $node_id
   *   The node object.
   *
   * @return array
   *   The HTML view mode.
   */
  protected function generateViewMode(int $node_id): array {
    return \Drupal::entityTypeManager()
      ->getViewBuilder('node')
      ->view(Node::load($node_id), $this->displayMode);
  }

  /**
   * Get related nodes.
   *
   * @return array
   *   The related nodes.
   */
  abstract protected function getRelateds(): array;

  /**
   * Get children menu entries for a parent menu link content sorted by weight.
   *
   * @param string $parent_menu_link_content_uuid
   *   The parent menu link content UUID.
   *
   * @return array
   *   The children menu entries.
   */
  protected function getChildrenMenuEntries(
    string $parent_menu_link_content_uuid,
  ): array {
    return \Drupal::entityTypeManager()
      ->getStorage('menu_link_content')
      ->getQuery()
      ->accessCheck(TRUE)
      ->condition('menu_name', $this->menuName)
      ->condition('enabled', 1)
      ->condition('parent', $parent_menu_link_content_uuid)
      ->sort('weight', 'ASC')
      ->execute();
  }

  /**
   * Get the menu entry for a given menu link content UUID.
   *
   * @param string $menu_link_content_uuid
   *   The menu link content UUID.
   *
   * @return int
   *   The menu entry.
   */
  protected function getMenuEntry(string $menu_link_content_uuid): int {
    $uuid_components = explode(':', $menu_link_content_uuid);
    $bundle = $uuid_components[0] ?? '';
    $uuid = $uuid_components[1] ?? '';

    $results = \Drupal::entityTypeManager()
      ->getStorage('menu_link_content')
      ->getQuery()
      ->accessCheck(TRUE)
      ->condition('bundle', $bundle)
      ->condition('uuid', $uuid)
      ->execute();

    return reset($results);
  }

  /**
   * Get the node ID to which an entity is finally attached.
   *
   * @param string $entity_id
   *   The entity ID.
   *
   * @return string
   *   The node ID.
   */
  public static function getNodeIdOfEntity(string $entity_id): string {
    $entity = \Drupal::entityTypeManager()
      ->getStorage('paragraph')
      ->load($entity_id);

    return $entity->get('parent_id')->value;
  }

}
