<?php

namespace Drupal\menu_family\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\menu_family\Enum\FamilyRelation;

/**
 * Provides a configuration form for the Menu Family module.
 */
class MenuFamilyConfigForm extends ConfigFormBase {

  /**
   * The entity display repository service.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityDisplayRepository = $container->get(
      'entity_display.repository'
    );
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['menu_family.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'menu_family_config';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('menu_family.settings');

    $display_modes = $this->entityDisplayRepository->getViewModes('node');

    $options = [];
    foreach ($display_modes as $key => $mode) {
      $options[$key] = $mode['label'];
    }

    $form['siblings'] = [
      '#type' => 'select',
      '#title' => $this->t('Display mode used for siblings pages'),
      '#options' => $options,
      '#default_value' => $config->get(
        FamilyRelation::Siblings->displayModeName()
      ),
    ];

    $form['children'] = [
      '#type' => 'select',
      '#title' => $this->t('Display mode used for children pages'),
      '#options' => $options,
      '#default_value' => $config->get(
        FamilyRelation::Children->displayModeName()
      ),
    ];

    $form['previous_next'] = [
      '#type' => 'select',
      '#title' => $this->t('Display mode used for previous and next pages'),
      '#options' => $options,
      '#default_value' => $config->get(
        FamilyRelation::PreviousNext->displayModeName()
      ),
    ];

    $menus = $this->entityTypeManager->getStorage('menu')->loadMultiple();
    $menu_options = [];

    foreach ($menus as $menu_id => $menu) {
      $menu_options[$menu_id] = $menu->label();
    }

    $form['menu'] = [
      '#type' => 'select',
      '#title' => $this->t('Menu used to search for related nodes'),
      '#options' => $menu_options,
      '#default_value' => $config->get('menu'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('menu_family.settings')
      ->set(
        FamilyRelation::Siblings->displayModeName(),
        $form_state->getValue('siblings'))
      ->set(
        FamilyRelation::Children->displayModeName(),
        $form_state->getValue('children'))
      ->set(
        FamilyRelation::PreviousNext->displayModeName(),
        $form_state->getValue('previous_next'))
      ->set(
        'menu',
        $form_state->getValue('menu'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
