<?php

namespace Drupal\menu_level_permission\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\ConfigTarget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\RedundantEditableConfigNamesTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for configuring menu level permission settings.
 *
 * @package Drupal\menu_level_permission\Form
 */
class MenuLevelPermissionSettingsForm extends ConfigFormBase {
  use RedundantEditableConfigNamesTrait;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'menu_level_permission.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'menu_level_permission_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $menu_list = array_map(function ($menu) {
      return $menu->label();
    }, $this->entityTypeManager->getStorage('menu')->loadMultiple());
    asort($menu_list);
    $form['restricted_menus'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Menus with restricted levels'),
      '#options' => $menu_list,
      '#config_target' => new ConfigTarget(
        'menu_level_permission.settings',
        'restricted_menus',
      ),
      '#description' => $this->t('Select menus for which you want to apply level permissions.'),
    ];
    $form['restricted_levels'] = [
      '#type' => 'select',
      '#title' => $this->t('Restricted menu levels'),
      '#options' => [
        1 => $this->t('Level 1 Only'),
        2 => $this->t('Levels 1-2'),
        3 => $this->t('Levels 1-3'),
        4 => $this->t('Levels 1-4'),
        5 => $this->t('Levels 1-5'),
      ],
      '#config_target' => new ConfigTarget(
        'menu_level_permission.settings',
        'restricted_levels',
      ),
      '#description' => $this->t('Select how many menu levels are permission restricted.'),
    ];

    return parent::buildForm($form, $form_state);
  }

}
