<?php

namespace Drupal\menu_level_permission\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\menu_level_permission\Access\MenuLevelPermissionAccess;
use Drupal\menu_link_content\MenuLinkContentInterface;
use Drupal\node\NodeInterface;
use Drupal\system\MenuInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class for validating menu item creation/modification.
 */
class MenuLevelPermissionFormValidator {
  use StringTranslationTrait;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $userAccount;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * The menu level permission access service.
   *
   * @var \Drupal\menu_level_permission\Access\MenuLevelPermissionAccess
   */
  protected MenuLevelPermissionAccess $menuLevelPermissionAccess;

  /**
   * Class constructor.
   *
   * @param \Drupal\Core\Session\AccountInterface $userAccount
   *   The "current_user" service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The "entity_type.manager" service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param \Drupal\menu_level_permission\Access\MenuLevelPermissionAccess $menuLevelPermissionAccess
   *   The menu level permission access service.
   */
  public function __construct(AccountInterface $userAccount, EntityTypeManagerInterface $entityTypeManager, ConfigFactoryInterface $config_factory, MenuLevelPermissionAccess $menuLevelPermissionAccess) {
    $this->userAccount = $userAccount;
    $this->entityTypeManager = $entityTypeManager;
    $this->config = $config_factory->get('menu_level_permission.settings');
    $this->menuLevelPermissionAccess = $menuLevelPermissionAccess;
  }

  /**
   * Provides dependency injection support.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
      $container->get('menu_level_permission.access')
    );
  }

  /**
   * Performs menu edit form validation.
   *
   * @param array $form
   *   The node form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Object containing user-submitted data.
   */
  public function validate(array $form, FormStateInterface $form_state): void {
    // Check if the user has permission to administer restricted menu levels.
    if ($this->userAccount->hasPermission('administer restricted menu levels')) {
      return;
    }

    /** @var \Drupal\Core\Entity\EntityFormInterface $entityForm */
    $entityForm = $form_state->getFormObject();

    // Get the restricted menus from configuration.
    $restricted_menus = $this->config->get('restricted_menus');
    // Get the restricted levels from configuration.
    $restricted_levels = $this->config->get('restricted_levels');

    if (!empty($restricted_menus) && $entity = $entityForm->getEntity()) {
      // A node or menu link is being saved. Check the selected parent.
      if ($entity instanceof NodeInterface || $entity instanceof MenuLinkContentInterface) {
        $input = $form_state->getUserInput();
        $values = $form_state->getValues();
        // Menu parent typically has a default value.
        // If editing an existing menu link, entity_id will be set.
        // If making a new menu link, entity_id = 0 and title will be set.
        // Enabled could be unchecked if removing an existing menu item.
        if ($entity instanceof NodeInterface && isset($input['menu']['menu_parent']) &&
          (!empty($input['menu']['title']) || $values['menu']['entity_id'] > 0)) {
          $menu_parent = $input['menu']['menu_parent'];
        }
        elseif ($entity instanceof MenuLinkContentInterface && isset($input['menu_parent'])) {
          $menu_parent = $input['menu_parent'];
        }
        else {
          // The user wasn't allowed or didn't enter menu details.
          // Remove menu details from the form state.
          if ($entity instanceof NodeInterface &&
            isset($values['menu']['menu_parent'])) {
            unset($values['menu']);
            $form_state->setValues($values);
          }
          return;
        }

        $parent_parts = explode(':', $menu_parent);
        if (isset($parent_parts[0])) {
          $menu = $parent_parts[0];
          if (!isset($restricted_menus[$menu]) || !$restricted_menus[$menu]) {
            // If the menu is not restricted, no need to validate.
            return;
          }

          // Check the level of the selected parent.
          // Default to parent level 0.
          $parent_level = 0;
          if (isset($parent_parts[1]) && $parent_parts[1] === 'menu_link_content' && isset($parent_parts[2])) {
            $parent_uuid = $parent_parts[2];
            $parent_level = $this->menuLevelPermissionAccess->getMenuLinkLevel($parent_uuid);
          }
          if ($parent_level < $restricted_levels) {
            // If the parent level is not at or deeper than the restricted
            // level, error.
            $form_state->setErrorByName('menu][menu_parent', $this->t('You must select a parent at level @level or deeper.', ['@level' => $restricted_levels]));
          }
        }
      }
      // This is the menu edit form where users can bulk edit menu link parent,
      // weight, and enabled values.
      elseif ($entity instanceof MenuInterface) {
        // Get the links being saved.
        $input = $form_state->getUserInput();
        $values = $form_state->getValues();
        if (isset($input['links'])) {
          // Check each menu link in the form.
          foreach ($input['links'] as $key => $link) {
            // Load the editable links from the form storage.
            $form_storage = $form_state->getStorage();
            $editable_links = $form_storage['editable_links'] ?? [];
            if (empty($editable_links)) {
              // This should not be possible but just in case.
              $form_state->setErrorByName('links][' . $link['id'], $this->t('This form does not contain editable menu links.'));
            }
            if (!in_array($key, $editable_links)) {
              // Remove un-editable links from form before submitting.
              unset($input['links'][$key]);
              unset($values['links'][$key]);
            }
            // This is an allowed link to edit.
            // Check parent is at or deeper than the restricted level.
            elseif (isset($link['parent']) && empty($link['parent'])) {
              // If the parent is empty, it means the link is at the top level.
              // Level 1 is always restricted if the menu is restricted.
              $form_state->setErrorByName('links][' . $link['id'], $this->t('You may only select a parent at @level or deeper.', ['@level' => ($restricted_levels)]));
            }
            elseif (isset($link['parent'])) {
              $parent_parts = explode(':', $link['parent']);
              if (isset($parent_parts[0]) && $parent_parts[0] === 'menu_link_content' && isset($parent_parts[1])) {
                $parent_uuid = $parent_parts[1];
                $parent_level = $this->menuLevelPermissionAccess->getMenuLinkLevel($parent_uuid);
                if ($parent_level < $restricted_levels) {
                  // Error if the parent is not at or deeper than the
                  // restricted level.
                  $form_state->setErrorByName('links][' . $link['id'], $this->t('You may only select a parent at @level or deeper.', ['@level' => ($restricted_levels)]));
                }
              }
            }
          }
          // Update the user input so it no longer includes restricted links.
          $form_state->setUserInput($input);
          $form_state->setValues($values);
        }
      }
    }
  }

}
