<?php

namespace Drupal\menu_level_permission\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Checks access for displaying administer menu pages.
 */
class MenuLevelPermissionAccess implements ContainerInjectionInterface {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * Creates a new MenuLevelPermissionAccess instance.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   */
  public function __construct(ModuleHandlerInterface $module_handler, EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory) {
    $this->moduleHandler = $module_handler;
    $this->entityTypeManager = $entity_type_manager;
    $this->config = $config_factory->get('menu_level_permission.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler'),
      $container->get('entity_type.manager'),
      $container->get('config.factory')
    );
  }

  /**
   * Overwrite menu link content access.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account to check access for.
   * @param \Drupal\menu_link_content\Entity\MenuLinkContent|null $menu_link_content
   *   The menu link content entity to check access for.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   The access result.
   */
  public function menuItemAccess(AccountInterface $account, ?MenuLinkContent $menu_link_content = NULL): AccessResult {
    if (!$menu_link_content instanceof MenuLinkContent) {
      return AccessResult::neutral();
    }
    // Check if user has the permission to administer restricted menu levels.
    if ($account->hasPermission('administer restricted menu levels')) {
      return $this->accessFallback($account, $menu_link_content);
    }

    // Get the menu of the menu_link_content.
    $menu = $menu_link_content->get('menu_name')->getString();
    if (empty($menu)) {
      return $this->accessFallback($account, $menu_link_content);
    }

    // Check if this is a restricted menu.
    $restricted_menus = $this->config->get('restricted_menus');
    if (!isset($restricted_menus[$menu]) || !$restricted_menus[$menu]) {
      return $this->accessFallback($account, $menu_link_content);
    }

    // Check if this is a restricted menu level.
    $restricted_levels = $this->config->get('restricted_levels');
    $menu_level = $this->getMenuLinkLevel($menu_link_content->uuid());

    // If the menu level is not higher than the restriction level, do not allow.
    if (is_numeric($restricted_levels) && $menu_level <= $restricted_levels) {
      return AccessResult::forbidden();
    }

    return $this->accessFallback($account, $menu_link_content);
  }

  /**
   * Fallback access result.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account to check access for.
   * @param \Drupal\menu_link_content\Entity\MenuLinkContent|null $menu_link_content
   *   The menu link content entity to check access for.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   The access result.
   */
  private function accessFallback(AccountInterface $account, ?MenuLinkContent $menu_link_content = NULL): AccessResult {
    // If the menu_admin_per_menu module is enabled, fallback to if the user has
    // permission to administer this menu.
    if ($this->moduleHandler->moduleExists('menu_admin_per_menu')) {
      // Avoid a dependency on menu_admin_per_menu.
      // @phpcs:disable
      // @phpstan-ignore-next-line
      $allowedMenusService = \Drupal::service('menu_admin_per_menu.allowed_menus');
      // @phpcs:enable
      return $allowedMenusService->menuItemAccess($account, $menu_link_content);
    }
    // Fallback to menu link content access.
    elseif ($account->hasPermission('administer menu')) {
      return AccessResult::allowed();
    }
    else {
      return AccessResult::neutral();
    }
  }

  /**
   * Get the level of a menu link content.
   *
   * @param string $uuid
   *   Unique identifier of the menu link content.
   *
   * @return int
   *   The menu level.
   */
  public function getMenuLinkLevel(string $uuid): int {
    $level = 1;
    while (TRUE) {
      $parent = $this->findMenuParent($uuid);
      if ($parent) {
        $level++;
        $uuid = $parent;
      }
      else {
        break;
      }
    }

    return $level;
  }

  /**
   * Helper function for finding menu parents.
   *
   * @param string $uuid
   *   Unique identifier of the menu link content.
   *
   * @return false|string
   *   The UUID of the parent menu link content, or FALSE if no parent is found.
   */
  protected function findMenuParent(string $uuid): false|string {
    // Load menu link content storage.
    $storage = $this->entityTypeManager->getStorage('menu_link_content');
    if ($items = $storage->loadByProperties(['uuid' => $uuid])) {
      $item = reset($items);
      $parent_id = $item->getParentId();
      if ($parent_id) {
        $parts = explode(':', $parent_id);
        if (isset($parts[1])) {
          return $parts[1];
        }
      }
    }
    // If no parent found, return FALSE.
    return FALSE;
  }

}
