<?php

/**
 * @file
 * Theme functions for the menu_link_content_revisions_ui module.
 */

use Drupal\Core\Url;

/**
 * Prepares variables for menu link content templates.
 *
 * This is intended to provide a basic display when viewing revisions.
 *
 * @param array $variables
 *   An associative array containing:
 *   - menu_link_content: An individual menu_link_content for display.
 */
function menu_link_content_revisions_ui_preprocess_menu_link_content(array &$variables): void {
  /** @var \Drupal\menu_link_content\Entity\MenuLinkContent $menu_link_content */
  $menu_link_content = $variables['elements']['#menu_link_content'];
  $variables['label'] = $menu_link_content->getTitle();
  $variables['attributes']['class'][] = 'menu-link-content';

  $menu_name = $menu_link_content->getMenuName();
  $variables['elements']['menu_name'] = [
    '#markup' => '<div class="menu-link-content__menu">' . t('<div class="field__label">Menu</div> @menu_name', ['@menu_name' => $menu_name]) . '</div>',
  ];

  // Add link information.
  $url = $menu_link_content->getUrlObject();
  if ($url->isRouted()) {
    if ($url->getRouteName() === '<nolink>') {
      $variables['elements']['url'] = [
        '#prefix' => '<div class="menu-link-content__url">',
        '#markup' => t('<div class="field__label">Path</div> none'),
        '#suffix' => '</div>',
      ];
    }
    else {
      $variables['elements']['url'] = [
        '#prefix' => '<div class="menu-link-content__url">',
        '#type' => 'link',
        '#title' => t('<div class="field__label">Path</div> @path', ['@path' => $url->toString()]),
        '#url' => $url,
        '#suffix' => '</div>',
      ];
    }
  }

  // Add parent menu item information.
  $pid = $menu_link_content->getParentId();
  if ($pid && str_starts_with($pid, 'menu_link_content:')) {
    // Load the parent menu link content entity.
    $uuid = substr($pid, 18);
    $parent_items = \Drupal::entityTypeManager()->getStorage('menu_link_content')->loadByProperties(['uuid' => $uuid]);
    foreach ($parent_items as $parent_item) {
      $parent_title = t('<div class="field__label">Parent Menu Item</div> @parent_title', ['@parent_title' => $parent_item->getTitle()]);
      $parent_link = $parent_item->get('link')->getValue();
      if (isset($parent_link[0]['uri'])) {
        $variables['elements']['parent'] = [
          '#prefix' => '<div class="menu-link-content__parent">',
          '#type' => 'link',
          '#title' => $parent_title,
          '#url' => Url::fromUri($parent_link[0]['uri']),
          '#suffix' => '</div>',
        ];
      }
      else {
        $variables['elements']['parent'] = [
          '#markup' => '<div class="menu-link-content__parent">' . $parent_title . '</div>',
        ];
      }
    }
  }
  else {
    $variables['elements']['parent'] = [
      '#markup' => '<div class="menu-link-content__parent">' . t('<div class="field__label">Parent</div> None; top level link') . '</div>',
    ];
  }
}
