<?php

namespace Drupal\Tests\menu_link_content_revisions_ui\Functional;

use Drupal\Core\Url;
use Drupal\menu_link_content\MenuLinkContentInterface;
use Drupal\menu_link_content\Plugin\Menu\MenuLinkContent;
use Drupal\Tests\BrowserTestBase;

/**
 * Test revision log page.
 *
 * @group menu_link_content_revisions_ui
 */
class MenuLinkContentRevisionsUiLogPageTest extends BrowserTestBase {

  /**
   * User with admin privileges.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * The local tasks block.
   *
   * @var \Drupal\block\Entity\Block
   */
  protected $tasksBlock;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The default theme.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * The modules to load to run the test.
   *
   * @var array
   */
  protected static $modules = ['block', 'user', 'node', 'menu_ui', 'menu_link_content_revisions_ui'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = \Drupal::service('entity_type.manager');

    // Add local task block.
    $this->tasksBlock = $this->drupalPlaceBlock('local_tasks_block', [
      'id' => 'tabs_block',
    ]);

    // Create a menu.
    $menuStorage = $this->entityTypeManager->getStorage('menu');
    $menu = $menuStorage->create([
      'id' => 'test',
      'label' => 'Test',
    ]);
    $menu->save();

    // Create node type with test menu as an available menu.
    $nodeType = $this->entityTypeManager->getStorage('node_type')->create([
      'type' => 'mypage',
      'name' => 'My Page',
      'available_menus' => ['test'],
      'parent' => 'test:',
    ]);
    $nodeType->save();

    // Create admin user.
    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
      'administer menu',
      'administer nodes',
      'create mypage content',
      'edit any mypage content',
      'view all menu link content revisions',
    ]);

    // Login.
    $this->drupalLogin($this->adminUser);
  }

  /**
   * Test enabling automatic menu link content revisions.
   */
  protected function testSettingsForm() {
    $web_assert = $this->assertSession();

    // Navigate to the settings form.
    $settingsFormPath = Url::fromRoute('menu_link_content_revisions_ui.settings');
    $this->drupalGet($settingsFormPath);

    // Assure we loaded the settings form.
    $web_assert->statusCodeEquals(200);

    // Enable automatic revisions.
    $this->submitForm([
      'enable_auto_revisions' => TRUE,
    ], 'Save configuration');

    $web_assert->pageTextContains('The configuration options have been saved.');
  }

  /**
   * Test that revisions log local task and operation link exist.
   */
  public function testWithoutAutoRevisions() {
    // Create a menu link.
    $menuLinkStorage = $this->entityTypeManager->getStorage('menu_link_content');
    $menuLinkHome = $menuLinkStorage->create([
      'title' => 'Home',
      'link' => ['uri' => 'internal:/'],
      'menu_name' => 'test',
      'weight' => -1,
    ]);
    $menuLinkHome->save();

    $web_assert = $this->assertSession();

    // Navigate to the menu form.
    $menuForm = Url::fromRoute('entity.menu.edit_form', ['menu' => 'test']);
    $this->drupalGet($menuForm);

    // Assure we loaded the menu form.
    $web_assert->statusCodeEquals(200);

    // Click the revisions operation link on the menu link.
    $this->clickLink('Revisions');

    // Confirm that a revision page loaded.
    $web_assert->pageTextContains('Current revision');

    // Go back to the edit tab.
    $this->clickLink('Edit');

    // Update the menu link.
    $this->submitForm([
      'title[0][value]' => 'Home updated',
    ], 'Save');

    $web_assert->pageTextContains('The menu link has been saved.');

    // Visit the revisions tab.
    $this->clickLink('Revisions');

    // Should still only be a single revision.
    $rows = $this->xpath('//table/tbody/tr');
    $this->assertCount(1, $rows, 'The table still only contains a single revision.');
  }

  /**
   * Test menu administration with auto revisions.
   */
  public function testMenuAdminWithAutoRevisions() {
    // Enable automatic revisions.
    $this->testSettingsForm();

    $web_assert = $this->assertSession();

    // Navigate to the add menu link form.
    $newMenuLinkForm = Url::fromRoute('entity.menu.add_link_form', ['menu' => 'test']);
    $this->drupalGet($newMenuLinkForm);

    // Assure we loaded the new menu link form.
    $web_assert->statusCodeEquals(200);

    // Create a new enabled menu link.
    $this->submitForm([
      'title[0][value]' => 'Front page',
      'link[0][uri]' => '<front>',
    ], 'Save');

    $web_assert->pageTextContains('The menu link has been saved.');

    // Get menu link content ID from the URL.
    $currentUrl = $this->getPath($this->getUrl());
    // Parse the URL to get the path segments.
    $pathSegments = explode('/', parse_url($currentUrl, PHP_URL_PATH));
    // Menu link ID should be the 5th segment admin/structure/menu/item/[id].
    $menuLinkId = $pathSegments[4];

    // Load the menu link.
    $menuLink = $this->entityTypeManager->getStorage('menu_link_content')->load($menuLinkId);
    $this->assertInstanceOf(MenuLinkContentInterface::class, $menuLink, 'The created link should be an instance of MenuLinkContentInterface.');

    // Visit the revisions tab.
    $this->clickLink('Revisions');

    // Confirm that a revision with the configured log message was created.
    $web_assert->pageTextContains('Added via menu link administration form (enabled)');

    // Go back to the edit tab.
    $this->clickLink('Edit');

    // Update the menu link.
    $this->submitForm([
      'title[0][value]' => 'Front page updated',
    ], 'Save');

    $web_assert->pageTextContains('The menu link has been saved.');

    // Visit the revisions tab.
    $this->clickLink('Revisions');

    // Confirm that the initial revision is still present.
    $web_assert->pageTextContains('Added via menu link administration form (enabled)');

    // Confirm that a second revision was created.
    $web_assert->pageTextContains('Updated via menu link administration form (enabled)');

    // Navigate to the menu form.
    $menuForm = Url::fromRoute('entity.menu.edit_form', ['menu' => 'test']);
    $this->drupalGet($menuForm);

    // Disable the menu link in the menu form.
    $this->submitForm([
      'links[menu_plugin_id:menu_link_content:' . $menuLink->uuid() . '][enabled]' => FALSE,
    ], 'Save');

    $web_assert->pageTextContains('Menu Test has been updated.');

    // Click the revisions operation link on the menu link.
    $this->clickLink('Revisions');

    // Confirm that revision with the correct message was created.
    $web_assert->pageTextContains('Order/status changed via menu administration form (disabled)');
  }

  /**
   * Test node form with menu links + auto revisions.
   */
  public function testNodesWithAutoRevisions() {
    // Enable automatic revisions.
    $this->testSettingsForm();

    $web_assert = $this->assertSession();

    // Navigate to mypage node add form.
    $newNodeForm = Url::fromRoute('node.add', ['node_type' => 'mypage']);
    $this->drupalGet($newNodeForm);

    // Assure we loaded the add node form.
    $web_assert->statusCodeEquals(200);

    // Add a mypage node.
    $this->submitForm([
      'title[0][value]' => 'Test Page',
      'menu[enabled]' => TRUE,
      'menu[title]' => 'Test Page',
    ], 'Save');

    $web_assert->pageTextContains('My Page Test Page has been created.');

    // Get the node ID from the url.
    $currentUrl = $this->getPath($this->getUrl());
    // Parse the URL to get the path segments.
    $pathSegments = explode('/', parse_url($currentUrl, PHP_URL_PATH));
    // Node ID should be the 2nd segment node/[id].
    $nodeId = $pathSegments[1];

    // Load menu links that point to the canonical route of the node.
    /** @var \Drupal\Core\Menu\MenuLinkManagerInterface $menuLinkManager */
    $menuLinkManager = \Drupal::service('plugin.manager.menu.link');
    $menuLinks = $menuLinkManager->loadLinksByRoute(
      'entity.node.canonical',
      ['node' => $nodeId]
    );

    // Get the first menu link from the results.
    $menuLink = reset($menuLinks);
    $this->assertInstanceOf(MenuLinkContent::class, $menuLink, 'The node should be associated with an instance of MenuLinkContent.');

    $menuLinkContent = $menuLink->getEntity();
    $menuLinkContentId = $menuLinkContent->id();

    // Navigate to the menu link content revisions page.
    $menuLinkRevisions = Url::fromRoute('entity.menu_link_content.version_history', ['menu_link_content' => $menuLinkContentId]);
    $this->drupalGet($menuLinkRevisions);

    $web_assert->pageTextContains('Added via node form (enabled)');

    // Edit the previous node.
    $editNodeForm = Url::fromRoute('entity.node.edit_form', ['node' => $nodeId]);
    $this->drupalGet($editNodeForm);

    // Assure we loaded the edit node form.
    $web_assert->statusCodeEquals(200);

    // Update node without updating the menu link.
    $this->submitForm([
      'title[0][value]' => 'Test Page Updated',
    ], 'Save');

    $web_assert->pageTextContains('My Page Test Page Updated has been updated.');

    $this->drupalGet($menuLinkRevisions);

    // Make sure the earlier revision is present.
    $web_assert->pageTextContains('Added via node form (enabled)');

    // Make sure a new revision was NOT created.
    $web_assert->pageTextNotContains('Updated via node form');

    // Edit again and edit the menu title.
    $this->drupalGet($editNodeForm);

    $this->submitForm([
      'menu[title]' => 'Test Page Updated',
    ], 'Save');

    $web_assert->pageTextContains('My Page Test Page Updated has been updated.');

    $this->drupalGet($menuLinkRevisions);

    // Make sure the earlier revision is present.
    $web_assert->pageTextContains('Added via node form (enabled)');

    // Make sure a new revision WAS created.
    $web_assert->pageTextContains('Updated via node form');
  }

  /**
   * Make sure we have just the path of the page.
   *
   * @param string $currentUrl
   *   Current url.
   *
   * @return string
   *   Current path.
   */
  private function getPath(string $currentUrl): string {
    $base_url = $this->baseUrl;
    $currentUrl = str_replace($base_url, '', $currentUrl);
    return ltrim($currentUrl, '/');
  }

}
