<?php

namespace Drupal\menu_markdown_token\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Menu\MenuLinkTreeInterface;
use Drupal\Core\Url;

/**
 * Service for generating markdown from menu hierarchies.
 */
class MenuMarkdownGenerator {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The menu link tree service.
   *
   * @var \Drupal\Core\Menu\MenuLinkTreeInterface
   */
  protected $menuLinkTree;

  /**
   * Constructs a MenuMarkdownGenerator object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Menu\MenuLinkTreeInterface $menu_link_tree
   *   The menu link tree service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, MenuLinkTreeInterface $menu_link_tree) {
    $this->entityTypeManager = $entity_type_manager;
    $this->menuLinkTree = $menu_link_tree;
  }

  /**
   * Generates markdown output for a menu.
   *
   * @param string $menu_name
   *   The menu machine name.
   * @param int $max_depth
   *   The maximum depth to traverse. Defaults to 10.
   *
   * @return string
   *   The markdown representation of the menu.
   */
  public function generate(string $menu_name, int $max_depth = 10): string {
    $tree = $this->buildMenuTree($menu_name, $max_depth);
    return $this->convertTreeToMarkdown($tree);
  }

  /**
   * Builds the menu tree.
   *
   * @param string $menu_name
   *   The menu machine name.
   * @param int $max_depth
   *   The maximum depth to traverse.
   *
   * @return array
   *   The menu tree.
   */
  protected function buildMenuTree(string $menu_name, int $max_depth): array {
    $parameters = $this->menuLinkTree->getCurrentRouteMenuTreeParameters($menu_name);

    // Set max depth from parameter.
    $parameters->setMaxDepth($max_depth);

    $tree = $this->menuLinkTree->load($menu_name, $parameters);

    // Apply menu link tree manipulators.
    $manipulators = [
      ['callable' => 'menu.default_tree_manipulators:checkAccess'],
      ['callable' => 'menu.default_tree_manipulators:generateIndexAndSort'],
    ];

    $tree = $this->menuLinkTree->transform($tree, $manipulators);

    return $tree;
  }

  /**
   * Converts menu tree to markdown format.
   *
   * @param array $tree
   *   The menu tree.
   * @param int $depth
   *   The current depth level.
   *
   * @return string
   *   The markdown representation.
   */
  protected function convertTreeToMarkdown(array $tree, int $depth = 0): string {
    $markdown = '';
    // Use 4 spaces per level for better markdown compatibility.
    $indent = str_repeat('    ', $depth);

    foreach ($tree as $element) {
      if (!$element->link || !$element->link->isEnabled()) {
        continue;
      }

      $title = $element->link->getTitle();
      $url = $this->generateUrl($element->link->getUrlObject());

      if ($url) {
        $markdown .= $indent . '- [' . $title . '](' . $url . ')' . "\n";
      }
      else {
        // For items without URLs (like <nolink>).
        $markdown .= $indent . '- ' . $title . "\n";
      }

      // Process children recursively.
      if ($element->hasChildren && !empty($element->subtree)) {
        $markdown .= $this->convertTreeToMarkdown($element->subtree, $depth + 1);
      }
    }

    return $markdown;
  }

  /**
   * Generates URL from a URL object.
   *
   * @param \Drupal\Core\Url $url_object
   *   The URL object.
   *
   * @return string|null
   *   The generated URL or NULL if not routed.
   */
  protected function generateUrl(Url $url_object): ?string {
    try {
      if ($url_object->isRouted()) {
        // Handle special routes.
        if ($url_object->getRouteName() === '<front>') {
          return '/';
        }
        if ($url_object->getRouteName() === '<nolink>' || $url_object->getRouteName() === '<button>') {
          return NULL;
        }
        return $url_object->toString();
      }
      elseif ($url_object->isExternal()) {
        return $url_object->toString();
      }
    }
    catch (\Exception $e) {
      // If URL generation fails, return NULL.
      return NULL;
    }

    return NULL;
  }

}
