/*
 * Extract Drupal "Parent Link" dropdown into multiple cascading dropdowns
 *
 * Original code by Jim Keller 2018
 * Eastern Standard
 * https://www.easternstandard.com
 */

/**
 * Get the value of an object property or false if it does not exist.
 *
 * @param obj
 * @param path
 * @returns {*|boolean}
 */
function getNestedProperty(obj, path) {
  if (!obj || typeof obj !== 'object' || path.length === 0) {
    return false;
  }

  let current = obj;
  for (let i = 0; i < path.length; i++) {
    const key = path[i];
    // Use optional chaining to safely access nested properties
    current = current?.[key];
    if (current === undefined && i < path.length - 1) {
      // If an intermediate property is undefined, the full path cannot exist
      return false;
    }
  }
  if (current !== undefined) {
    return current;
  }
  return false;
}

function SelectExtractor(settings) {
  settings = settings || {};
  this.settings = settings;

  // Get the original select box.
  const selectBoxes = document.querySelectorAll(
    this.setting('selectors.select_input_original').toString(),
  );
  let selectBoxId = '';
  let selectBoxExtractedHide = false;

  this.optionData = {};
  this.optionDataTop = {};
  this.selectBoxes = [];
  this.currentActiveTrail = this.setting('selectors.active_set');
  this.currentMenuParentId = '';
  this.currentSelectValue = '';

  this.debug(`|${this.setting('selectors.select_input_original')}|`);
  this.debug(selectBoxes);

  if (selectBoxes.length > 0) {
    for (let i = 0; i < selectBoxes.length; i++) {
      selectBoxId = `se_${SelectExtractor.unique_index_get().toString()}`;
      selectBoxes[i].setAttribute('data-select-extractor-id', selectBoxId);

      // Extract the options from the original select.
      this.select_options_extract(selectBoxes[i]);

      // Hide the original select box.
      if (this.setting('select_input_original_hide')) {
        this.select_box_original_hide(selectBoxes[i]);
      }

      if (selectBoxes[i].selectedIndex >= 0) {
        this.currentSelectValue =
          selectBoxes[i][selectBoxes[i].selectedIndex].value;
        this.currentMenuParentId = this.currentSelectValue.substring(
          0,
          this.currentSelectValue.indexOf(':') + 1,
        );
        this.select_box_change_link_initialize(selectBoxes[i]);
        selectBoxExtractedHide = true;
      }
    }
    this.select_box_create(this.optionDataTop, 0, selectBoxId);
  } else {
    this.debug('No matching selectors found for SelectExtractor');
  }
}

SelectExtractor.prototype.select_box_change_link_initialize = function (
  selectBox,
) {
  const selectBoxContainer = document.querySelector(
    this.setting('selectors.select_box_container').toString(),
  );
  const selectBoxId = selectBox.getAttribute('data-select-extractor-id');
  const selectionLabel = document.createElement('div');

  selectionLabel.innerHTML = `Currently Selected Parent: ${selectBox.options[
    selectBox.selectedIndex
  ].text.replace(/[-<>]/g, '')}`;
  selectionLabel.classList.add(
    'select-box-label-current',
    'form-item__description',
  );
  selectionLabel.setAttribute('data-select-original-id', selectBoxId);

  selectBoxContainer.appendChild(selectionLabel);
};

/* Hide the original select box. */
SelectExtractor.prototype.select_box_original_hide = function (selectBox) {
  selectBox.style.height = '0';
  selectBox.style.width = '0';
  selectBox.style.lineHeight = '0';
  selectBox.style.visibility = 'hidden';
  selectBox.style.minHeight = 0;
  selectBox.style.padding = 0;
  selectBox.style.position = 'absolute';
};

SelectExtractor.prototype.in_active_trail = function (option) {
  const id = option.substring(option.indexOf(':') + 1, option.length);
  for (let i = 0; i < Object.keys(this.currentActiveTrail).length; i++) {
    if (this.currentActiveTrail[id]) {
      return true;
    }
  }
  return false;
};

SelectExtractor.prototype.select_box_create = function (
  optionData,
  level,
  selectOriginalId,
  methodOptions,
) {
  methodOptions = methodOptions || {};

  const newSelect = document.createElement('select');
  const me = this;
  let optionCount = 0;
  let wrapper;

  newSelect.classList.add('form-element--type-select', 'form-element');
  newSelect.appendChild(
    new Option(
      this.setting('option_empty_label').toString(),
      this.setting('option_empty_value').toString(),
    ),
  );
  const optionDataKeys = Object.keys(optionData);
  optionDataKeys.forEach((key) => {
    if (
      (this.currentMenuParentId && key === this.currentMenuParentId) ||
      optionData[key].value === this.currentSelectValue ||
      this.in_active_trail(optionData[key].value)
    ) {
      newSelect.appendChild(
        new Option(optionData[key].text, optionData[key].value, false, true),
      );
    } else {
      newSelect.appendChild(
        new Option(optionData[key].text, optionData[key].value),
      );
    }
    optionCount++;
  });

  newSelect.setAttribute('data-select-level', level);
  newSelect.setAttribute('data-select-original-id', selectOriginalId);
  newSelect.setAttribute('id', `${selectOriginalId}_${level.toString()}`);

  if (
    typeof methodOptions.hide !== 'undefined' &&
    methodOptions.hide === true
  ) {
    newSelect.style.display = 'none';
  }

  const event = new Event('change');

  newSelect.addEventListener('change', function () {
    me.select_handle_change(newSelect);
  });

  this.selectBoxes.push(newSelect);
  const selectBoxContainer = document.querySelector(
    this.setting('selectors.select_box_container').toString(),
  );

  if (this.setting('select_box_wrapper.element')) {
    wrapper = document.createElement(
      this.setting('select_box_wrapper.element').toString(),
    );

    if (this.setting('select_box_wrapper.class')) {
      wrapper.classList.add(
        this.setting('select_box_wrapper.class').toString(),
      );
    }

    wrapper.appendChild(newSelect);
  } else {
    wrapper = newSelect;
  }

  selectBoxContainer.appendChild(wrapper);
  if (newSelect.selectedIndex > 0) {
    newSelect.dispatchEvent(event);
  }
};

SelectExtractor.prototype.select_handle_change = function (sourceElement) {
  this.select_option_apply(
    sourceElement,
    sourceElement.options[sourceElement.selectedIndex].value,
  );
};

SelectExtractor.prototype.select_option_apply = function (
  sourceElement,
  optionVal,
) {
  this.select_visibility_refresh(sourceElement);

  if (
    typeof this.optionData[optionVal] !== 'undefined' &&
    typeof this.optionData[optionVal].children !== 'undefined'
  ) {
    this.select_box_create(
      this.optionData[optionVal].children,
      parseInt(sourceElement.getAttribute('data-select-level'), 10) + 1,
      sourceElement.getAttribute('data-select-original-id'),
    );
  }

  this.select_original_sync(sourceElement);
};

SelectExtractor.prototype.select_original_sync = function (sourceElement) {
  const selectBoxes = this.select_boxes_get_all_for_original_id(
    sourceElement.getAttribute('data-select-original-id'),
  );
  let maxLevel = -1;
  let deepestIndex = -1;
  let deepestSelect = null;
  let selectedValue = null;
  let originalSelect = null;
  let curLevel = null;

  // Find the deepest select box that has a value.
  for (let i = 0; i < selectBoxes.length; i++) {
    curLevel = selectBoxes[i].getAttribute('data-select-level');

    selectedValue = selectBoxes[i].value;

    if (
      curLevel > maxLevel &&
      selectedValue !== this.setting('option_empty_value').toString()
    ) {
      maxLevel = curLevel;
      deepestIndex = i;
    }
  }

  // Set original select to value of the deepest child box that has a selection.
  if (deepestIndex > -1) {
    deepestSelect = selectBoxes[deepestIndex];
    originalSelect = this.original_select_by_child_select(deepestSelect);

    originalSelect.value = deepestSelect.value;
  }
};

SelectExtractor.prototype.original_select_by_child_select = function (element) {
  const originalId = element.getAttribute('data-select-original-id');
  return document.querySelector(
    `select[data-select-extractor-id="${originalId}"]`,
  );
};

SelectExtractor.prototype.select_boxes_get_all_for_original_id = function (
  originalId,
) {
  return document.querySelectorAll(
    `select[data-select-original-id="${originalId}"]`,
  );
};

SelectExtractor.prototype.select_visibility_refresh = function (sourceElement) {
  let curLevel = null;
  const sourceLevel = sourceElement.getAttribute('data-select-level');
  const selectOriginalId = sourceElement.getAttribute(
    'data-select-original-id',
  );

  const selectBoxes =
    this.select_boxes_get_all_for_original_id(selectOriginalId);

  for (let i = 0; i < selectBoxes.length; i++) {
    curLevel = selectBoxes[i].getAttribute('data-select-level');

    // Remove any select boxes that are dependent on the value of this one.
    if (curLevel > sourceLevel) {
      selectBoxes[i].parentNode.removeChild(selectBoxes[i]);
    }
  }
};

SelectExtractor.prototype.select_options_extract = function (element) {
  const parents = [];
  let curParentOption = null;
  let curLevel = 0;
  let optionLevel = null;
  let nextParentOption = null;
  let thisOptionData = {};

  for (let i = 0; i < element.options.length; i++) {
    optionLevel = this.select_option_determine_level_by_text(
      element.options[i].text,
    );

    this.debug(`CHECKING: ${element.options[i].text} at curLevel ${curLevel}`);

    if (optionLevel === 0) {
      // Top level option.

      this.debug('option level zero');

      thisOptionData = {
        text: element.options[i].text,
        value: element.options[i].value,
        children: [],
      };

      // Store the top level data, so we can seed the first dropdown.
      this.optionDataTop[element.options[i].value] = thisOptionData;

      nextParentOption = element.options[i];

      this.debug('setting cur level to zero');
      curLevel = 0;
    } else {
      // Hit a new level.
      if (curLevel !== optionLevel) {
        if (optionLevel > curLevel) {
          // We found a new level of depth.
          this.debug(
            `option level of ${optionLevel.toString()} > cur level of ${curLevel.toString()} for ${
              element.options[i].text
            }`,
          );
          if (curParentOption) {
            this.debug(` cur parent is ${curParentOption.text}`);
          }
          if (nextParentOption) {
            this.debug(` next parent is ${nextParentOption.text}`);
          }

          if (nextParentOption) {
            // An item only becomes a parent if we find something beneath it.
            // We stored 'next parent option' on the previous loop iteration.
            // If we find a child beneath it, then it becomes a parent.
            if (
              typeof this.optionData[nextParentOption.value] === 'undefined'
            ) {
              this.debug(
                `  adding ${element.options[i].text} to option data array`,
              );
              this.optionData[nextParentOption.value] = {
                text: nextParentOption.text,
                value: nextParentOption.value,
                children: [],
              };

              curParentOption = nextParentOption;
              this.debug(`  pushing to parents array: ${curParentOption.text}`);
              parents.push(curParentOption);
            }
          }

          // Item now "on deck" to become a parent if we find a child beneath it.
          nextParentOption = element.options[i];

          curLevel++;
        } else {
          // Lower level
          this.debug(
            ` option level of ${optionLevel.toString()} < cur level of ${curLevel.toString()} for ${
              element.options[i].text
            }`,
          );

          // Retreat back to the last parent we had at this level.
          while (curLevel > optionLevel) {
            curParentOption = parents.pop();
            curLevel--;
          }

          this.debug(`new cur level is ${curLevel}`);

          // Set this item up as the next potential parent,
          // and also set cur_parent to the deepest item in the parents array.
          nextParentOption = element.options[i];
          curParentOption = parents[parents.length - 1];

          if (curParentOption) {
            this.debug(` popped parent is ${curParentOption.text}`);
          }
        }
      } else {
        this.debug(
          ` option level of ${optionLevel} is the same as curLevel of ${
            curLevel
          }`,
        );

        nextParentOption = element.options[i];
      }

      if (curParentOption) {
        this.debug(
          `adding ${element.options[i].text} as child of ${
            curParentOption.text
          }`,
        );

        this.optionData[curParentOption.value].children.push({
          text: element.options[i].text,
          value: element.options[i].value,
        });
      }
    }
  }
};

SelectExtractor.prototype.select_option_determine_level_by_text = function (
  optionText,
) {
  let level = 0;
  const prefix = this.setting('option_level_prefix').toString();
  let substrStart = 0;
  let substrEnd = substrStart + prefix.length;

  while (optionText.substring(substrStart, substrEnd) === prefix) {
    level++;
    substrStart += prefix.length;
    substrEnd += prefix.length;
  }

  return level;
};

/**
 * Gets a setting by key name. Check local setting; fall back to global setting.
 * @param {string} key
 */
SelectExtractor.prototype.setting = function (key) {
  let value = false;
  const keyArray = key.split('.');
  let nestedValue = getNestedProperty(this.settings, keyArray);
  // Found a property with a local value.
  if (nestedValue) {
    value = nestedValue;
  } else {
    // Check the default settings.
    const defaultSettings = SelectExtractor.settings_default();
    nestedValue = getNestedProperty(defaultSettings, keyArray);
    // Found a property with a default value.
    if (nestedValue) {
      value = nestedValue;
    }
  }
  return value;
};

SelectExtractor.prototype.debug = function (...args) {
  if (this.setting('debug')) {
    if (args.length > 0) {
      for (let i = 0; i < args.length; i++) {
        if (
          typeof console !== 'undefined' &&
          typeof console.log !== 'undefined'
        ) {
          console.log(args[i]);
        }
      }
    }
  }
};

SelectExtractor.log = function (...args) {
  if (args.length > 0) {
    for (let i = 0; i < args.length; i++) {
      if (
        typeof console !== 'undefined' &&
        typeof console.log !== 'undefined'
      ) {
        console.log(args[i]);
      }
    }
  }
};

SelectExtractor.settings_default = function () {
  return {
    selectors: {
      select_input_original: '#edit-menu-parent',
      select_box_container: '.form-item--menu-parent',
    },
    select_input_original_hide: true,
    option_level_prefix: '--',
    option_empty_label: '-Choose-',
    option_empty_value: '_none',
    debug: false,
    select_box_wrapper: {
      element: 'div',
      class: 'select-box-extracted__wrapper',
    },
  };
};

/**
 * Set up unique indexes so this script can be called on multiple select boxes
 * without getting confused.
 */
SelectExtractor.unique_index = 0;
SelectExtractor.unique_index_get = function () {
  return SelectExtractor.unique_index++;
};
