<?php

namespace Drupal\Tests\menu_parent_form_ui\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Test form element.
 *
 * @group menu_parent_form_ui
 */
class MenuParentFormUiElementTest extends WebDriverTestBase {

  /**
   * The default theme.
   *
   * @var string
   */
  protected $defaultTheme = 'claro';

  /**
   * The installation profile.
   *
   * @var string
   */
  protected $profile = 'standard';

  /**
   * User with admin privileges.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * Maximum menu depth.
   *
   * @var int
   */
  protected $maxDepth;

  /**
   * The modules to load to run the test.
   *
   * @var array
   */
  protected static $modules = ['user', 'node', 'menu_parent_form_ui'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Store the max depth of the tree.
    $this->maxDepth = \Drupal::service('menu.link_tree')->maxDepth();

    // Create admin user.
    $this->adminUser = $this->drupalCreateUser([
      'administer menu',
      'administer nodes',
      'create page content',
      'edit any page content',
    ]);

    // Login.
    $this->drupalLogin($this->adminUser);
  }

  /**
   * Test menu link form.
   */
  public function testMenuLinkForm() {
    $last_link_id = NULL;
    $plugin_ids = [];
    $page = $this->getSession()->getPage();

    // Create a maximum number of menu links, each a child of the previous.
    for ($i = 0; $i <= $this->maxDepth - 1; $i++) {
      $this->drupalGet('admin/structure/menu/manage/main/add');

      // For each previously created menu level, select parent level option.
      for ($j = 0; $j <= $i; $j++) {
        $selected = $this->checkHierarchicalSelects($plugin_ids, $j);
        $page->selectFieldOption('se_0_' . $j, $selected);
      }

      // Fill other required fields and submit menu link form.
      $title = 'title' . $i;
      $page->fillField('link[0][uri]', '/');
      $page->fillField('title[0][value]', $title);
      $this->click('#edit-submit');

      // Save the created menu link for later use as a parent.
      $menu_links = \Drupal::entityTypeManager()->getStorage('menu_link_content')->loadByProperties(['title' => $title]);
      $last_link = reset($menu_links);
      $plugin_ids[] = $last_link->getPluginId();
      $last_link_id = $last_link->id();
    }
    // Edit the last menu item.
    $this->drupalGet('admin/structure/menu/item/' . $last_link_id . '/edit');

    // Confirm that the hidden parent element has the correct value.
    // This would be the second to last menu link created.
    $last_parent = 'main:' . $plugin_ids[count($plugin_ids) - 2];
    $this->assertSession()->fieldExists('edit-menu-parent');
    $this->assertSession()->fieldValueEquals('edit-menu-parent', $last_parent);

    // Confirm that the hierarchical selects have the correct values.
    for ($j = 0; $j <= $this->maxDepth - 1; $j++) {
      $this->assertSession()->fieldExists('se_0_' . $j);
      if ($j < 1) {
        $selected = 'main:';
      }
      else {
        $selected = 'main:' . $plugin_ids[$j - 1];
      }
      $this->assertSession()->fieldValueEquals('se_0_' . $j, $selected);
    }

    // The last link cannot be a parent in the new menu link form.
    $this->drupalGet('admin/structure/menu/manage/main/add');

    // For each previously created menu level, select parent level option.
    for ($j = 0; $j <= $this->maxDepth; $j++) {
      // Cannot choose a parent at the max depth.
      if ($j === $this->maxDepth) {
        $this->assertSession()->assertNoElementAfterWait('css', '#se_0_' . $j);
      }
      // Select the next level of parent.
      else {
        $selected = $this->checkHierarchicalSelects($plugin_ids, $j);
        $page->selectFieldOption('se_0_' . $j, $selected);
      }
    }
  }

  /**
   * Test the node edit form.
   */
  public function testNodeForm() {
    $last_node = NULL;
    $plugin_ids = [];
    $page = $this->getSession()->getPage();

    // Add nodes with menu links, each a child of the previous.
    for ($i = 0; $i <= $this->maxDepth - 1; $i++) {
      $this->drupalGet('node/add/page');

      // Fill node title field.
      $title = 'node_title_' . $i;
      $page->fillField('title[0][value]', $title);

      // Expand menu details.
      $this->click('[aria-controls="edit-menu"]');
      // Enable menu item on node form.
      // The menu link title will default to the node title.
      $page->checkField('edit-menu-enabled');

      // For each previously created menu level, select parent level option.
      for ($j = 0; $j <= $i; $j++) {
        $selected = $this->checkHierarchicalSelects($plugin_ids, $j);
        $page->selectFieldOption('se_0_' . $j, $selected);
      }

      // Submit node form.
      $this->click('#edit-submit');

      // Save the created menu link for later use as a parent.
      $menu_links = \Drupal::entityTypeManager()->getStorage('menu_link_content')->loadByProperties(['title' => $title]);
      $last_link = reset($menu_links);
      $plugin_ids[] = $last_link->getPluginId();

      // Save the last node for later use.
      if ($i === ($this->maxDepth - 1)) {
        $nodes = \Drupal::entityTypeManager()->getStorage('node')->loadByProperties(['title' => $title]);
        $last_node = reset($nodes);
      }
    }

    // Edit the last node.
    $this->drupalGet('node/' . $last_node->id() . '/edit');

    // Confirm that the hidden parent element has the correct value.
    // This would be the second to last node created.
    $last_parent = 'main:' . $plugin_ids[count($plugin_ids) - 2];
    $this->assertSession()->fieldExists('edit-menu-menu-parent');
    $this->assertSession()->fieldValueEquals('edit-menu-menu-parent', $last_parent);

    // Confirm that the hierarchical selects have the correct values.
    for ($j = 0; $j <= $this->maxDepth - 1; $j++) {
      $this->assertSession()->fieldExists('se_0_' . $j);
      if ($j < 1) {
        $selected = 'main:';
      }
      else {
        $selected = 'main:' . $plugin_ids[$j - 1];
      }
      $this->assertSession()->fieldValueEquals('se_0_' . $j, $selected);
    }
  }

  /**
   * Check hierarchical select values.
   */
  protected function checkHierarchicalSelects(array $plugin_ids, int $j): string {
    if ($j < 1) {
      $selected = 'main:';
    }
    else {
      $selected = 'main:' . $plugin_ids[$j - 1];
    }
    $this->assertSession()->waitForField('se_0_' . $j);
    $this->assertSession()->fieldExists('se_0_' . $j);
    $this->assertSession()->optionExists('se_0_' . $j, $selected);
    return $selected;
  }

}
