<?php

namespace Drupal\menu_synergy\Drush\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\menu_synergy\Service\SyncMenuBatchService;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A Drush commandfile.
 */
final class MenuSynergyCommands extends DrushCommands {

  /**
   * Constructs a MenuSynergyCommands object.
   *
   * @param \Drupal\menu_synergy\Service\SyncMenuBatchService $batchService
   *   Batch service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   */
  public function __construct(
    private readonly SyncMenuBatchService $batchService,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('menu_synergy.sync_menu_batch_service'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Update internal menu item links to be stored as entity links.
   */
  #[CLI\Command(name: 'menu_synergy:sync_menu', aliases: ['msm'])]
  #[CLI\Argument(name: 'menu', description: 'Machine name of the menu to sync.')]
  #[CLI\Argument(name: 'batchSize', description: 'Maximum menu items to update per batch.')]
  #[CLI\Argument(name: 'limit', description: 'Maximum menu items to update in a single operation.')]
  #[CLI\Option(name: 'include-disabled', description: 'Include disabled menu items.')]
  #[CLI\Usage(name: 'menu_synergy:sync_menu main 20 100', description: 'Update first 100 enabled internal menu items in the main menu in batches of 20.')]
  #[CLI\Usage(name: 'menu_synergy:sync_menu main --include-disabled', description: 'Update all internal menu items in the main menu including disabled items.')]
  public function syncMenu(?string $menu = NULL, ?int $batchSize = NULL, ?int $limit = NULL, array $options = [
    'include-disabled' => FALSE
  ]): void {

    // The menu argument is required.
    if ($menu === NULL) {
      throw new \InvalidArgumentException(dt('The menu argument is required.'));
    }

    // Make sure the menu exists.
    if (!$this->entityTypeManager
      ->getStorage('menu')
      ->getQuery()
      ->condition('id', $menu)
      ->range(0, 1)
      ->count()
      ->execute()) {
      throw new \InvalidArgumentException(
        dt("The menu '!menu' does not exist.", ['!menu' => $menu])
      );
    }
    $includeDisabled = FALSE;
    if ($options['include-disabled'] === TRUE) {
      $includeDisabled = TRUE;
    }
    if ($batchSize === NULL) {
      $batchSize = 20;
    }

    $this->batchService->create($menu, $batchSize, $limit, $includeDisabled);
  }

}
