<?php

namespace Drupal\menu_synergy\Service;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Psr\Log\LoggerInterface;

/**
 * Service for creating batches.
 */
class SyncMenuBatchService {

  use StringTranslationTrait;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $loggerChannel;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $connection;

  /**
   * Inject services.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\mysql\Driver\Database\mysql\Connection $connection
   *   The database connection.
   */
  public function __construct(
    LoggerChannelFactoryInterface $logger_factory,
    Connection $connection
  ) {
    $this->loggerChannel = $logger_factory->get('menu_synergy');
    $this->connection = $connection;
  }

  /**
   * {@inheritdoc}
   */
  public function create(string $menu, int $batchSize = 20, ?int $limit = NULL, bool $includeDisabled = FALSE): void {
    $batchBuilder = (new BatchBuilder())
      ->setTitle($this->t('Update internal menu items to be stored as entity links.'))
      ->setFinishCallback([self::class, 'finishProcess'])
      ->setInitMessage('Initializing')
      ->setProgressMessage('Completed @current of @total.');

    $this->addOperations($batchBuilder, $menu, $batchSize, $limit, $includeDisabled);

    batch_set($batchBuilder->toArray());
    $this->loggerChannel->notice('Batch created.');
    if (PHP_SAPI === 'cli' && function_exists('drush_backend_batch_process')) {
      drush_backend_batch_process();
    }

  }

  /**
   * {@inheritdoc}
   */
  public static function processItems($itemsToProcess, $batchSize, $total, &$context): void {

    // Process elements stored in each batch (operation).
    foreach ($itemsToProcess as $item) {
      $context['results'][] = $item;
      self::updateMenuItem($item);
    }

    // Message displayed above the progress bar or in the CLI.
    $processedItems = !empty($context['results']) ? count($context['results']) : $batchSize;
    $context['message'] = 'Checked ' . $processedItems . '/' . $total . ' Items';

    \Drupal::logger('menu_synergy')->notice('Batch processing completed: Updated ' . $processedItems . '/' . $total . ' Items');
  }

  /**
   * {@inheritdoc}
   */
  public static function finishProcess($success, $results, array $operations): void {

    // Do something when processing is finished.
    if ($success) {
      \Drupal::logger('menu_synergy')->notice('Batch processing completed.');
    }
    if (!empty($operations)) {
      \Drupal::logger('menu_synergy')->error('Batch processing failed: ' . implode(', ', $operations));
    }
  }

  /**
   * Add operations to the Batch.
   *
   * @param \Drupal\Core\Batch\BatchBuilder $batchBuilder
   *   Reference to the batchBuilder object.
   * @param string $menu
   *   Machine name of the menu to sync.
   * @param int $batchSize
   *   Size of the batch to be operated on.
   * @param int|null $limit
   *   Maximum number of menu items to process.
   * @param bool $includeDisabled
   *   Include disabled menu items.
   */
  private function addOperations(BatchBuilder &$batchBuilder, string $menu, int $batchSize, ?int $limit, bool $includeDisabled): void {
    // We are using a static query for performance reasons.
    $sql = "SELECT id FROM {menu_link_content_data} WHERE menu_name = :menu_name AND link__uri LIKE 'internal:/node/%' ";
    if (!$includeDisabled) {
      $sql .= "AND enabled = 1 ";
    }
    $sql .= "AND SUBSTRING(link__uri, 16) NOT IN (SELECT SUBSTRING(link__uri, 13) FROM {menu_link_content_data} WHERE menu_name = :menu_name AND link__uri LIKE 'entity:node/%') ";
    $sql .= "AND link__uri NOT IN (SELECT link__uri FROM {menu_link_content_data} WHERE menu_name = :menu_name AND link__uri LIKE 'internal:/node/%' GROUP BY link__uri HAVING COUNT(link__uri) > 1) ORDER BY id ASC";
    $params = [
      ':menu_name' => $menu,
    ];
    // Limit the number of nodes to process.
    // Static queries don't support more than one placeholder.
    if ($limit) {
      $sql .= " LIMIT " . $limit;
    }
    $results = $this->connection->query($sql, $params);
    $ids = $results->fetchCol();

    $total = count($ids);
    $itemsToProcess = [];
    $i = 0;

    // Create multiple batch operations based on the $batchSize.
    foreach ($ids as $id) {
      $i++;
      $itemsToProcess[] = $id;
      if ($i == $total || !($i % $batchSize)) {

        $batchBuilder->addOperation(
          [
            self::class,
            'processItems',
          ],
          [
            $itemsToProcess,
            $batchSize,
            $total,
          ]
        );

        $itemsToProcess = [];
      }
    }

  }

  /**
   * Update internal menu links to entity links.
   */
  private static function updateMenuItem(int $id): void {
    /** @var \Drupal\Core\Entity\RevisionableStorageInterface $nodeStorage */
    $menuLinkStorage = \Drupal::service('entity_type.manager')
      ->getStorage('menu_link_content');

    $menuLink = $menuLinkStorage->load($id);
    $url = $menuLink->getUrlObject();
    if ($url instanceof Url) {
      if (!$url->isRouted() || !isset($url->getRouteParameters()['node'])) {
        \Drupal::logger('menu_synergy')->notice('Menu link ' . $id . ' references an invalid node.');
        return;
      }
      $nid = $url->getRouteParameters()['node'];
      $menuLink->set('link', ['uri' => 'entity:node/' . $nid]);
      $menuLink->save();
      \Drupal::logger('menu_synergy')->notice('Menu link ' . $id . ' for node ' . $nid . ' updated to an entity link.');
    }

  }

}
