/**
 * Create a dynamic React component from an imported Twig file.
 */

import HTMLReactParser from 'html-react-parser';
import { createMarkup } from 'twing';
import TwingEnvironment from '../../../.storybook/twingEnvironment.js'
import { useEffect, useState, isValidElement } from 'react';

/**
 * Recursive function to identify all React elements and set up later
 * replacements for them, since React elements can't go directly into the
 * rendering of a Twig template.
 * @param {*}     context           The variable currently being scanned for React elements.
 * @param {Array} interactiveValues A list of all React elements that have
 *                                  been found so far.
 * @return {Array} First item is the context, possibly with React components
 *   replaced with temporary divs. Second item is the current
 *   interactiveValues list.
 */
const collectInteractiveValues = ( context, interactiveValues = [] ) => {
	// If a React element, set it up for later replacement after Twig rendering,
	// then immediately return before we try to handle regular objects.
	if ( isValidElement( context ) ) {
		interactiveValues.push( context );
		context = createMarkup( `<span class="replace-twig-variable" data-index="${ interactiveValues.length - 1 }"></span>`, 'utf-8' );
		return [ context, interactiveValues ];
	}

	// Iterate over properties of regular objects.
	if ( context?.constructor === Object ) {
		for ( const [ key, value ] of Object.entries( context ) ) {
			[ context[ [ key ] ], interactiveValues ] = collectInteractiveValues( value, interactiveValues );
		}
	}

	// Iterate over array items.
	if ( context?.constructor === Array ) {
		for ( let i = 0; i < context.length; i++ ) {
			[ context[ i ], interactiveValues ] = collectInteractiveValues( context[ i ], interactiveValues );
		}
	}

	return [ context, interactiveValues ];
};

/**
 * The main component.
 *
 * @param {Object}   context              Variables for the Twig template,
 *                                        possibly including React components.
 * @param {Function} context.twigFunction Function returning a promise,
 *                                        generated by importing a Twig file with Twing.
 * @return {React.FunctionComponent}      A React component to display the rendered Twig file.
 */
export default ( { twigFunction, ...context } ) => {
	const [ output, setOutput ] = useState( '' );

	const [ staticContext, interactiveValues ] = collectInteractiveValues( { ...context } );

	// Because Twing rendering is asynchronous, we have to use hooks and state to
	// update the content. This is also why these always have to be dynamic blocks
	// that render via PHP, instead of using the normal block editor Save
	// function. (Perhaps it's for the best, though, since it means we use PHP
	// Twig instead of Twing for the final rendering.)
	useEffect(
		() => {
			// Twing imports Twig files as a function that returns a promise.
			twigFunction.render( TwingEnvironment, staticContext ).then( ( newOutput ) => {
				setOutput(
					// Hydrate the rendered Twig file into React elements.
					HTMLReactParser(
						newOutput,
						{
							// Replace our temporary static divs with the original React
							// elements we set aside.
							replace: ( domNode ) => {
								if ( domNode.attribs && domNode.attribs.class === 'replace-twig-variable' ) {
									return interactiveValues[ domNode.attribs[ 'data-index' ] ];
								}
							},
						}
					)
				);
			} );
		},
		[ JSON.stringify( staticContext ), JSON.stringify( interactiveValues) ]
	);

	return output;
};
