# Mercury Editor Page Templates

A Drupal module that provides a template system for creating pages with pre-configured layouts and content using Mercury Editor.

## Overview

This module allows content creators to select from pre-defined templates when creating new pages, streamlining the page creation process and ensuring consistency across the site. Templates can be configured for specific content types and include complex layouts with multiple paragraphs and nested content.

## Features

- **Template Management**: Create, edit, and delete page templates through an admin interface
- **Content Type Integration**: Templates can be configured for specific content types
- **Drupal Permissions Integration**: Uses standard Drupal permissions for blank page creation
- **Completely Configurable**: No hardcoded values - everything is configurable through YAML
- **Flexible Field Structure**: Support for any field structure and naming convention
- **YAML Configuration**: Templates are defined using YAML for easy editing and version control
- **Mercury Editor Integration**: Seamless integration with Mercury Editor for page creation
- **Validation**: Ensures content types are properly configured in Mercury Editor before allowing template creation
- **Region Support**: Support for layout regions when placing content
- **Nested Paragraphs**: Support for complex nested paragraph structures

## Requirements

- Drupal 10/11
- Mercury Editor module
- Layout Paragraphs module
- Paragraphs module

## Installation

1. Place the module in `web/modules/custom/mercury_editor_page_templates/`
2. Enable the module: `drush en mercury_editor_page_templates`
3. Clear cache: `drush cache:rebuild`

## Configuration

### 1. Configure Mercury Editor

Before creating templates, ensure your content types are configured in Mercury Editor:

1. Go to **Configuration > Content authoring > Mercury Editor settings**
2. Configure the desired content types under the "Bundles" section
3. Save the configuration

### 2. Create Page Templates

1. Go to **Configuration > Content authoring > Mercury Editor Page Templates**
2. Configure global settings:
   - **Template Selector Theme**: Choose which theme to use for the template selector interface
   - **Blank Page Permissions**: Use Drupal's permission system to control who can create blank pages
3. Click "Add Template" to create a new template
4. Fill in the template details:
   - **Template Label**: Display name for the template
   - **Template Description**: Description of what the template contains
   - **Template Paragraphs (YAML)**: Define the template structure in YAML
   - **Available for Content Types**: Select which content types can use this template

### 3. Blank Page Permissions

The "Create Blank Page" option is controlled using Drupal's standard permission system:

1. Go to **People > Permissions**
2. Find the "Create blank pages without templates" permission
3. Assign this permission to the roles that should be able to create blank pages
4. Users without this permission will only see available templates (no blank page option)

### 5. YAML Template Structure

Templates are defined using YAML with complete configurability. Every aspect of the paragraph creation can be customized:

#### Basic Text Block Example

```yaml
-
  type: text_block
  content: 'Your content here'
  text_format: basic_html
  field_config:
    field_name: field_paragraph_text
    field_structure:
      value: value
      format: format
```

#### Custom Field Structure Example

```yaml
-
  type: custom_paragraph
  description: 'Your content here'
  field_config:
    field_name: field_custom_text
    field_structure:
      value: body
      format: full_html
```

#### Two Column Layout Example

```yaml
-
  type: custom_section
  description: 'Main Content Section'
  layout: two_column_75_25
  paragraphs:
    -
      type: text_block
      description: 'Text that appears in the first region'
      field_config:
        -
          field_name: body
          field_structure:
            value: 'Text that appears in the first region'
            format: full_html
      region: first
    -
      type: text_block
      description: 'Text that appears in the second region'
      field_config:
        -
          field_name: body
          field_structure:
            value: 'Text that appears in the second region'
            format: full_html
      region: second
```

#### Section with Multiple Field Paragraphs

```yaml
-
  type: custom_section
  description: 'Main Content Section'
  layout: one_column
  paragraphs:
    -
      type: link_listing
      description: 'Link listing for landing page'
      field_config:
        -
          field_name: field_title
          field_structure:
            value: 'Link listing for landing page'
        -
          field_name: field_type
          field_structure:
            value: 'list'
      region: first
```

#### Field Configuration

The system uses a `field_config` array approach for all field configurations:

**Single Field Example:**
```yaml
-
  type: text_block
  description: 'Simple text block'
  field_config:
    -
      field_name: field_paragraph_text
      field_structure:
        value: 'Your content here'
        format: 'basic_html'
```

**Multiple Fields Example:**
```yaml
-
  type: link_listing
  description: 'Link listing for landing page'
  field_config:
    -
      field_name: field_title
      field_structure:
        value: 'Manual listing for landing page'
    -
      field_name: field_type
      field_structure:
        value: 'list'
```

This approach makes it consistent whether you're configuring one field or many fields - you always use the `field_config` array.

#### Style Options Support

You can also configure style options for paragraphs using the `style_options` key. The system supports both label names and CSS class values:

```yaml
-
  type: text_block
  description: 'Styled text block'
  field_config:
    -
      field_name: field_paragraph_text
      field_structure:
        value: 'Your content here'
        format: 'full_html'
  style_options:
    background_color: 'blue'
    text_alignment: 'center'
    border_style: 'solid'
```

**Style Option Value Formats:**

You can use either the display label (recommended) or the CSS class:

```yaml
# Using display labels (recommended - more readable)
style_options:
  background_color: 'Blue'           # Blue background
  text_alignment: 'Center'           # Center-aligned text
  border_style: 'Solid'              # Solid border
  padding_size: 'Large'              # Large padding

# Using CSS classes (also works)
style_options:
  background_color: 'bg-primary'     # Blue background
  text_alignment: 'text-center'      # Center-aligned text
  border_style: 'border-solid'       # Solid border
  padding_size: 'p-4'                # Large padding
```

#### Complex Paragraph with Multiple Fields and Style Options

```yaml
-
  type: custom_content_block
  description: 'Content block with title, body, and image'
  field_config:
    -
      field_name: field_title
      field_structure:
        value: 'Welcome to Our Site'
    -
      field_name: field_body
      field_structure:
        value: 'This is the main content of the block.'
        format: 'full_html'
    -
      field_name: field_image
      field_structure:
        target_id: 123
        alt: 'Welcome image'
        title: 'Welcome'
  style_options:
    background_color: 'Light Gray'
    text_alignment: 'Center'
    padding_size: 'Large'
```


#### Configuration Options

Each paragraph in your template can specify:

- **`type`**: The paragraph type machine name
- **`description`**: Used for display purposes in the template preview
- **`field_config`**: Array of field configuration objects
  - Each object contains:
    - **`field_name`**: The field machine name on the paragraph
    - **`field_structure`**: Object defining the field data structure
- **`style_options`**: Style options configuration (optional)
  - Key-value pairs where the key is the style option name and the value is the option value
  - Supports both display labels (recommended) and CSS class values
  - Example: `background_color: 'Blue'` or `background_color: 'bg-primary'`

#### Field Structure Flexibility

The system supports any field structure you need:

```yaml
# Standard Drupal text field structure
field_structure:
  value: value
  format: format

# Custom field structure
field_structure:
  value: body
  format: text_format

# Another custom structure
field_structure:
  value: content
  format: format_type
```

This creates field data like:
```php
// Standard structure
['value' => 'content', 'format' => 'basic_html']

// Custom structure
['body' => 'content', 'text_format' => 'basic_html']
```

## Usage

### Creating Pages with Templates

1. Navigate to **Content > Add content > [Content Type]**
2. If templates are configured for that content type, you'll be redirected to the template selector
3. Choose from available templates or create a blank page
4. Click "Use This Template" to apply the template and open in Mercury Editor

### Template Selector

The template selector provides:
- Visual preview of available templates
- Template descriptions
- Option to create a blank page
- Direct integration with Mercury Editor

## API

### Functions

#### `mercury_editor_page_templates_get_templates($content_type = NULL)`

Retrieves configured templates, optionally filtered by content type.

**Parameters:**
- `$content_type` (string, optional): Content type machine name to filter by

**Returns:**
- Array of template configurations

**Example:**
```php
// Get all templates
$all_templates = mercury_editor_page_templates_get_templates();

// Get templates for specific content type
$page_templates = mercury_editor_page_templates_get_templates('page');
```

#### `mercury_editor_page_templates_can_create_blank_page($account = NULL)`

Checks if a user can create blank pages without using templates.

**Parameters:**
- `$account` (AccountInterface, optional): User account to check. If not provided, uses the current user.

**Returns:**
- Boolean: TRUE if the user can create blank pages, FALSE otherwise.

**Example:**
```php
// Check if current user can create blank pages
$can_create_blank = mercury_editor_page_templates_can_create_blank_page();

// Check if specific user can create blank pages
$user = \Drupal\user\Entity\User::load(123);
$can_create_blank = mercury_editor_page_templates_can_create_blank_page($user);
```

## Hooks

### `hook_menu_local_tasks_alter()`

The module implements this hook to conditionally show the page templates tab only for content types that have templates configured.

## File Structure

```
mercury_editor_page_templates/
├── config/
│   └── schema/
│       └── mercury_editor_page_templates.schema.yml
├── src/
│   ├── Controller/
│   │   ├── NodeAddRedirectController.php
│   │   └── TemplateLibraryController.php
│   ├── EventSubscriber/
│   │   └── NodeAddRedirectSubscriber.php
│   └── Form/
│       ├── TemplateSelectorForm.php
│       └── TemplateSettingsForm.php
├── mercury_editor_page_templates.info.yml
├── mercury_editor_page_templates.libraries.yml
├── mercury_editor_page_templates.links.task.yml
├── mercury_editor_page_templates.menu.yml
├── mercury_editor_page_templates.module
├── mercury_editor_page_templates.routing.yml
├── mercury_editor_page_templates.services.yml
└── README.md
```

## Configuration Files

### `mercury_editor_page_templates.settings.yml`

Stores the template configurations:

```yaml
templates:
  template_id:
    label: 'Template Name'
    description: 'Template description'
    paragraphs:
      # YAML paragraph definitions
    content_types:
      page: page
      article: article
```

## Troubleshooting

### Common Issues

1. **Templates not showing for content type**
   - Ensure the content type is configured in Mercury Editor settings
   - Check that the template is assigned to the correct content type

2. **YAML parsing errors**
   - Validate YAML syntax using an online YAML validator
   - Ensure proper indentation (2 spaces)
   - Check for missing colons after field names

3. **Field content not appearing**
   - Verify each `field_config[].field_name` matches the paragraph field
   - Check that the field exists on the paragraph type
   - Ensure the `field_config[].field_structure` keys match your field's expected structure
   - Verify all required configuration is present (no missing values)

4. **Configuration errors**
   - Ensure all required fields are specified: `type`, `description`, and `field_config`
   - Check that each `field_config[]` entry has `field_name` and `field_structure`
   - For style options: Use either display labels (e.g., `'Blue'`) or CSS classes (e.g., `'bg-primary'`)
   - Verify field structure values match your field's expected format
   - No fallback values are provided - all configuration must be explicit

5. **Region not working**
   - Verify the region exists in the layout definition
   - Check the layout configuration in the paragraph type
   - Ensure proper YAML syntax for the region field


### Debugging

Enable logging to see detailed information about template processing:

```php
// In settings.php
$config['system.logging']['error_level'] = 'verbose';
```

Check the watchdog logs for Mercury Editor Page Templates messages:

```bash
drush watchdog:show --type=mercury_editor_page_templates
```

## Development

### Adding Custom Fields

To add support for custom fields in templates:

1. Add the field to your paragraph type
2. Configure the field structure in your YAML template using `field_config`:
   ```yaml
   field_config:
     field_name: your_field_machine_name
     field_structure:
       value: your_value_key
       format: your_format_key
   ```
3. The module will use your exact configuration without any assumptions

### Extending Templates

Templates can be extended by:
- Adding new paragraph types
- Creating custom layouts
- Implementing custom field types
- Adding validation rules

## Support

For issues and feature requests, please create an issue in the project repository.

## License

This module is licensed under the same terms as Drupal core.
