<?php

namespace Drupal\mercury_editor_page_templates\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for template library functionality.
 */
class TemplateLibraryController extends ControllerBase {

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a TemplateLibraryController object.
   *
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   */
  public function __construct(FormBuilderInterface $form_builder, EntityTypeManagerInterface $entity_type_manager, LoggerChannelFactoryInterface $logger_factory, MessengerInterface $messenger) {
    $this->formBuilder = $form_builder;
    $this->entityTypeManager = $entity_type_manager;
    $this->loggerFactory = $logger_factory;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('form_builder'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory'),
      $container->get('messenger')
    );
  }

  /**
   * Shows the template selector form.
   *
   * @return array
   *   The template selector form render array.
   */
  public function showTemplateSelector() {
    return $this->formBuilder->getForm('Drupal\mercury_editor_page_templates\Form\TemplateSelectorForm');
  }

  /**
   * Shows the template selector form for a specific content type.
   *
   * @param \Drupal\node\Entity\NodeType $node_type
   *   The node type entity.
   *
   * @return array
   *   The template selector form render array.
   */
  public function showContentTypeTemplates($node_type) {
    // Get the content type machine name.
    $content_type = $node_type->id();

    // Get templates configured for this content type.
    $templates = mercury_editor_page_templates_get_templates($content_type);

    // If no templates are configured for this content type, show a message.
    if (empty($templates)) {
      $build = [
        '#markup' => $this->t('No page templates are configured for @content_type content type.', [
          '@content_type' => $node_type->label(),
        ]),
      ];
      return $build;
    }

    // Create a custom form that shows only templates for this content type.
    $form = $this->formBuilder->getForm('Drupal\mercury_editor_page_templates\Form\TemplateSelectorForm', $content_type);

    return $form;
  }

}
