<?php

namespace Drupal\mercury_editor_page_templates\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Render\RendererInterface;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\layout_paragraphs\LayoutParagraphsLayout;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Form for selecting a template before creating a new page.
 */
class TemplateSelectorForm extends FormBase {

  /**
   * The content type for which to show templates.
   *
   * @var string|null
   */
  protected $contentType;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a TemplateSelectorForm object.
   *
   * @param string|null $content_type
   *   The content type for which to show templates.
   * @param \Drupal\Core\Render\RendererInterface|null $renderer
   *   The renderer service.
   */
  public function __construct($content_type = NULL, ?RendererInterface $renderer = NULL) {
    $this->contentType = $content_type;
    $this->renderer = $renderer ?: \Drupal::service('renderer');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'mercury_editor_page_templates_template_selector_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $request = \Drupal::request();
    $content_type = $this->contentType ?: $request->query->get('content_type');

    // If still no content type, try to get it from the route parameter.
    if (empty($content_type)) {
      $node_type = \Drupal::routeMatch()->getParameter('node_type');
      if ($node_type) {
        $content_type = $node_type->id();
      }
    }

    // Store the content type in the class property for use in other methods.
    $this->contentType = $content_type;

    // If no content type specified, show error.
    if (empty($content_type)) {
      $form['no_content_type'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['no-content-type-message']],
        '#markup' => '<div class="no-content-type"><h3>' . $this->t('No Content Type Specified') . '</h3><p>' . $this->t('A content type must be specified to show available templates.') . '</p></div>',
      ];
      return $form;
    }

    // Get templates filtered by content type.
    $templates = mercury_editor_page_templates_get_templates($content_type);

    if (empty($templates)) {
      $form['no_templates'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['no-templates-message']],
        '#markup' => '<div class="no-templates"><h3>' . $this->t('No Templates Available') . '</h3><p>' . $this->t('No templates are configured for the @content_type content type. You can either create a blank page or configure templates first.', ['@content_type' => $content_type]) . '</p><p><a href="/admin/config/content/mercury-editor/templates" class="button button--primary">' . $this->t('Configure Templates') . '</a></p></div>',
      ];
      return $form;
    }

    $template_id = $request->query->get('template');

    // If a template is selected via URL parameter, process it immediately.
    if ($template_id && isset($templates[$template_id])) {
      \Drupal::logger('mercury_editor_page_templates')->info('Template selected via URL: @template_id', ['@template_id' => $template_id]);
      $this->processTemplateSelection($template_id, $templates[$template_id], $form_state);
      return $form;
    }

    $form['#attached']['library'][] = 'mercury_editor_page_templates/template_selector';

    // Get content type label.
    $content_type_entity = \Drupal::entityTypeManager()->getStorage('node_type')->load($content_type);
    $content_type_label = $content_type_entity ? $content_type_entity->label() : $content_type;

    $form['header'] = [
      '#markup' => '<div class="template-selector-header"><h1>' . $this->t('Choose a Template') . '</h1><p>' . $this->t('Select a template to start building your new @content_type, or create a blank page.', ['@content_type' => $content_type_label]) . '</p></div>',
    ];

    // Display available templates.
    if (!empty($templates)) {
      $form['templates'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['template-grid']],
      ];

      foreach ($templates as $id => $template) {
        $form['templates'][$id] = [
          '#type' => 'container',
          '#attributes' => [
            'class' => ['template-card'],
            'data-template-id' => $id,
          ],
          '#markup' => $this->buildTemplateCard($template, $id),
        ];
      }
    }

    // Only show blank page option if user has permission.
    if (\Drupal::currentUser()->hasPermission('create blank pages without templates')) {
      $form['blank_page_section'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['blank-page-section']],
      ];

      $form['blank_page_section']['card'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['blank-page-card']],
      ];

      $form['blank_page_section']['card']['content'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['blank-page-card__content']],
      ];

      $form['blank_page_section']['card']['content']['title'] = [
        '#markup' => '<h3>' . $this->t('Start from Scratch') . '</h3>',
      ];

      $form['blank_page_section']['card']['content']['description'] = [
        '#markup' => '<p>' . $this->t('Create a blank page and build it from the ground up.') . '</p>',
      ];

      $form['blank_page_section']['card']['content']['create_blank'] = [
        '#type' => 'submit',
        '#value' => $this->t('Create Blank Page'),
        '#attributes' => [
          'class' => ['button', 'button--secondary', 'create-blank-page'],
        ],
        '#submit' => ['::createBlankPage'],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {}

  /**
   * Processes template selection and creates the node.
   *
   * @param string $template_id
   *   The template ID.
   * @param array $template
   *   The template data.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function processTemplateSelection($template_id, array $template, FormStateInterface $form_state) {
    try {
      \Drupal::logger('mercury_editor_page_templates')->info('Processing template selection: @template_id', ['@template_id' => $template_id]);

      // Get the content type from the form context.
      $content_type = $this->contentType;

      if (empty($content_type)) {
        throw new \Exception('Content type must be specified to create a node from template.');
      }

      // Create a new node with the template applied.
      $node = \Drupal::entityTypeManager()->getStorage('node')->create([
        'type' => $content_type,
        'title' => $template['label'] . ' (Template Draft)',
        'status' => 0,
      ]);

      // Create all paragraphs and add them to the layout.
      $field_name = $template['paragraph_reference_field_name'] ?? '';
      if (empty($field_name)) {
        throw new \Exception('Paragraph reference field name not configured for this template. Please configure the paragraph reference field name in the template settings.');
      }
      $field = $node->get($field_name);
      $layout = new LayoutParagraphsLayout($field);

      if (!$layout) {
        throw new \Exception('Could not create LayoutParagraphsLayout from field');
      }

      foreach ($template['paragraphs'] as $paragraph_config) {
        try {
          $paragraph = $this->createParagraphFromConfig($paragraph_config);
          $layout->appendComponent($paragraph);

          \Drupal::logger('mercury_editor_page_templates')->info(
            'Added paragraph to layout: @type with UUID @uuid',
            [
              '@type' => $paragraph_config['type'],
              '@uuid' => $paragraph->uuid(),
            ]
          );

          // Handle nested paragraphs if they exist.
          if (isset($paragraph->nested_paragraphs) && is_array($paragraph->nested_paragraphs)) {
            // Get the layout information from behavior settings.
            $layout_id = $paragraph->getBehaviorSetting('layout_paragraphs', 'layout');
            $default_region = $this->getDefaultRegionForLayout($layout_id);

            foreach ($paragraph->nested_paragraphs as $index => $nested_paragraph) {
              // Determine which region to use for this nested paragraph.
              $target_region = $this->getTargetRegionForNestedParagraph($paragraph_config, $index, $layout_id, $default_region);

              // Add nested paragraph to section's region (insertIntoRegion).
              try {
                $layout->insertIntoRegion($paragraph->uuid(), $target_region, $nested_paragraph);

                \Drupal::logger('mercury_editor_page_templates')->info(
                  'Added nested paragraph to section region @region: @type with UUID @uuid',
                  [
                    '@region' => $target_region,
                    '@type' => $nested_paragraph->bundle(),
                    '@uuid' => $nested_paragraph->uuid(),
                  ]
                );
              }
              catch (\Exception $e) {
                // If insertIntoRegion fails,
                // fall back to adding to main layout.
                $layout->appendComponent($nested_paragraph);

                \Drupal::logger('mercury_editor_page_templates')->warning(
                  'Failed to insert into region @region, added to main layout: @error',
                  [
                    '@region' => $target_region,
                    '@error' => $e->getMessage(),
                  ]
                );
              }
            }
          }
        }
        catch (\Exception $e) {
          \Drupal::logger('mercury_editor_page_templates')->error('Error creating paragraph: @error', ['@error' => $e->getMessage()]);
        }
      }

      // Create a node in memory (not saved to database).
      $node = \Drupal::entityTypeManager()->getStorage('node')->create([
        'type' => $content_type,
        'title' => $template['label'] . ' (Template)',
        'status' => 0,
        $field_name => $layout->getParagraphsReferenceField()->getValue(),
      ]);

      // Store the node in Mercury Editor tempstore (not database).
      $mercury_tempstore = \Drupal::service('mercury_editor.tempstore_repository');
      $mercury_tempstore->set($node);

      // Check if we're in Mercury Editor context.
      $request = \Drupal::request();
      $is_mercury_editor_context = $request->query->has('mercury_editor');

      // Redirect to Mercury Editor with the node UUID.
      $redirect_url = Url::fromRoute('mercury_editor.editor', [
        'mercury_editor_entity' => $node->uuid(),
      ], [
        'query' => [
          'template' => $template_id,
          'template_applied' => 1,
        ],
      ]);

      \Drupal::logger('mercury_editor_page_templates')->info('Setting redirect to Mercury Editor: @url', ['@url' => $redirect_url->toString()]);

      $this->messenger()->addStatus(
        $this->t('Template "@template" will be applied to your new page. Review and save when ready.', [
          '@template' => $template['label'],
        ])
      );

      // Perform direct redirect since form state redirect
      // doesn't work in buildForm.
      $response = new RedirectResponse($redirect_url->toString());
      $response->send();
      exit;

    }
    catch (\Exception $e) {
      \Drupal::logger('mercury_editor_page_templates')->error(
        'Error in template selector: @error',
        ['@error' => $e->getMessage()]
      );
      $this->messenger()->addError(
        $this->t('Error applying template: @error', ['@error' => $e->getMessage()])
      );
    }
  }

  /**
   * Submit handler for skipping template selection.
   */
  public function skipTemplate(array &$form, FormStateInterface $form_state) {
    try {
      // Get the content type from the form context.
      $content_type = $this->contentType;

      if (empty($content_type)) {
        throw new \Exception('Content type must be specified to create a blank node.');
      }

      // Create a blank node in memory (not saved to database).
      $node = \Drupal::entityTypeManager()->getStorage('node')->create([
        'type' => $content_type,
        'title' => 'New Page',
        'status' => 0,
      ]);

      // Store the node in Mercury Editor tempstore (not database).
      $mercury_tempstore = \Drupal::service('mercury_editor.tempstore_repository');
      $mercury_tempstore->set($node);

      // Redirect to the new node in Mercury Editor.
      $redirect_url = Url::fromRoute(
        'mercury_editor.editor',
        ['mercury_editor_entity' => $node->uuid()]
      );

      $this->messenger()->addStatus($this->t('New blank page created. You can now start building your page.'));

      $form_state->setRedirectUrl($redirect_url);

    }
    catch (\Exception $e) {
      \Drupal::logger('mercury_editor_page_templates')->error(
        'Error creating blank page: @error',
        ['@error' => $e->getMessage()]
      );
      $this->messenger()->addError(
        $this->t('Error creating page: @error', ['@error' => $e->getMessage()])
      );
    }
  }

  /**
   * Submit handler for creating a blank page.
   */
  public function createBlankPage(array &$form, FormStateInterface $form_state) {
    try {
      // Get the content type from the form context.
      $content_type = $this->contentType;

      if (empty($content_type)) {
        throw new \Exception('Content type must be specified to create a blank page.');
      }

      // Create a blank node in memory (not saved to database).
      $node = \Drupal::entityTypeManager()->getStorage('node')->create([
        'type' => $content_type,
        'title' => 'New Page',
        'status' => 0,
      ]);

      // Store the node in Mercury Editor tempstore (not database).
      $mercury_tempstore = \Drupal::service('mercury_editor.tempstore_repository');
      $mercury_tempstore->set($node);

      // Redirect to the new node in Mercury Editor.
      $redirect_url = Url::fromRoute(
        'mercury_editor.editor',
        ['mercury_editor_entity' => $node->uuid()]
      );

      $this->messenger()->addStatus($this->t('New blank page created. You can now start building your page.'));

      $form_state->setRedirectUrl($redirect_url);

    }
    catch (\Exception $e) {
      \Drupal::logger('mercury_editor_page_templates')->error(
        'Error creating blank page: @error',
        ['@error' => $e->getMessage()]
      );
      $this->messenger()->addError(
        $this->t('Error creating page: @error', ['@error' => $e->getMessage()])
      );
    }
  }

  /**
   * Creates a paragraph from configuration data, handling nested paragraphs.
   *
   * @param array $paragraph_config
   *   The paragraph configuration.
   *
   * @return \Drupal\paragraphs\Entity\Paragraph
   *   The created paragraph.
   */
  protected function createParagraphFromConfig(array $paragraph_config) {
    $paragraph = Paragraph::create([
      'type' => $paragraph_config['type'],
    ]);

    // Set content based on paragraph type and configuration.
    $paragraph_type = $paragraph_config['type'];

    // Handle field configs - expect array of field configurations.
    $field_configs = $paragraph_config['field_config'] ?? [];

    // Process all field configurations.
    foreach ($field_configs as $field_config) {
      if (isset($field_config['field_name']) && isset($field_config['field_structure'])) {
        $field_name = $field_config['field_name'];
        $field_structure = $field_config['field_structure'];

        // Build the field data structure dynamically based on configuration.
        $field_data = $field_structure;

        $paragraph->set($field_name, $field_data);

        // Debug: Log the field content being set.
        \Drupal::logger('mercury_editor_page_templates')->info(
          'Setting @type content in field: @field with structure: @structure',
          [
            '@type' => $paragraph_type,
            '@field' => $field_name,
            '@structure' => print_r($field_data, TRUE),
          ]
        );
      }
    }

    // Set style options.
    if (isset($paragraph_config['style_options'])) {
      foreach ($paragraph_config['style_options'] as $style_option_key => $style_option_value) {
        $style_option_plugin = \Drupal::service('style_options.discovery')->getOptionDefinition($style_option_key);

        $plugin_id = $style_option_plugin['option_id'];
        $plugin_type = $style_option_plugin['plugin'];

        if ($plugin_type === 'css_class') {
          // First try to find by class (exact match)
          $option_index = array_search($style_option_value, array_column($style_option_plugin['options'], 'class'));

          // If not found by class, try to find by label (case-insensitive)
          if ($option_index === FALSE) {
            $labels = array_column($style_option_plugin['options'], 'label');
            $option_index = array_search(strtolower($style_option_value), array_map('strtolower', $labels));
          }

          if ($option_index !== FALSE) {
            $paragraph->setBehaviorSettings('style_options', [$plugin_id => [$plugin_type => $option_index]]);
          }
        }
      }
    }

    // Check if this paragraph type has Layout Paragraphs behavior enabled.
    $paragraph_type = \Drupal::entityTypeManager()->getStorage('paragraphs_type')->load($paragraph_config['type']);
    $has_layout_paragraphs = FALSE;
    if ($paragraph_type) {
      $enabled_behaviors = $paragraph_type->getEnabledBehaviorPlugins();
      $has_layout_paragraphs = isset($enabled_behaviors['layout_paragraphs']);
    }

    // Handle paragraphs with Layout Paragraphs behavior.
    if ($has_layout_paragraphs && isset($paragraph_config['layout'])) {
      $paragraph->setBehaviorSettings('layout_paragraphs', [
        'layout' => $paragraph_config['layout'],
      ]);

      \Drupal::logger('mercury_editor_page_templates')->info(
        'Created @type paragraph with layout @layout',
        [
          '@type' => $paragraph_config['type'],
          '@layout' => $paragraph_config['layout'],
        ]
      );

      // Handle nested paragraphs within Layout Paragraphs sections.
      if (isset($paragraph_config['paragraphs']) && is_array($paragraph_config['paragraphs'])) {
        // Store nested paragraphs for later processing.
        // We'll add them to the main layout after the section is created.
        $nested_paragraphs = [];
        foreach ($paragraph_config['paragraphs'] as $nested_config) {
          $nested_paragraph = $this->createParagraphFromConfig($nested_config);
          $nested_paragraphs[] = $nested_paragraph;

          // Debug: Log the nested paragraph content.
          if ($nested_paragraph->bundle() === 'text_block') {
            // Handle unified field configuration approach.
            $nested_field_configs = [];

            // Check for field_config.
            if (isset($nested_config['field_config'])) {
              $field_config = $nested_config['field_config'];

              // If it's an array of field configurations, use it directly.
              if (is_array($field_config) && isset($field_config[0])) {
                $nested_field_configs = $field_config;
              }
              // If it's a single field configuration object, convert to array.
              elseif (is_array($field_config) && isset($field_config['field_name'])) {
                $nested_field_configs = [$field_config];
              }
            }

            // Log content for all configured fields.
            foreach ($nested_field_configs as $field_config) {
              if (isset($field_config['field_name'])) {
                $field_name = $field_config['field_name'];
                $text_content = $nested_paragraph->get($field_name)->value;
                \Drupal::logger('mercury_editor_page_templates')->info(
                  'Nested text_block content in field @field: @content',
                  ['@field' => $field_name, '@content' => $text_content]
                );
              }
            }
          }
        }

        // Store the nested paragraphs on the section for later processing.
        $paragraph->nested_paragraphs = $nested_paragraphs;

        \Drupal::logger('mercury_editor_page_templates')->info(
          'Prepared @type paragraph with @count nested paragraphs',
          [
            '@type' => $paragraph_config['type'],
            '@count' => count($paragraph_config['paragraphs']),
          ]
        );
      }
    }

    return $paragraph;
  }

  /**
   * Gets the default region for a layout.
   *
   * @param string $layout_id
   *   The layout ID.
   *
   * @return string
   *   The default region name.
   */
  protected function getDefaultRegionForLayout($layout_id) {
    // Load the layout definition from the layouts.yml file.
    $layout_definition = \Drupal::service('plugin.manager.core.layout')
      ->getDefinition($layout_id);

    if ($layout_definition && $layout_definition->getDefaultRegion()) {
      return $layout_definition->getDefaultRegion();
    }

    // Fallback to 'main' if no default region is found.
    \Drupal::logger('mercury_editor_page_templates')->warning(
      'No default region found for layout @layout_id, using fallback',
      ['@layout_id' => $layout_id]
    );

    return 'main';
  }

  /**
   * Gets the target region for a nested paragraph.
   *
   * @param array $paragraph_config
   *   The paragraph configuration.
   * @param int $index
   *   The index of the nested paragraph.
   * @param string $layout_id
   *   The layout ID.
   * @param string $default_region
   *   The default region for the layout.
   *
   * @return string
   *   The target region name.
   */
  protected function getTargetRegionForNestedParagraph(array $paragraph_config, $index, $layout_id, $default_region) {
    // Check if the paragraph config specifies regions for nested paragraphs.
    if (isset($paragraph_config['regions']) && is_array($paragraph_config['regions'])) {
      // Use the region specified for this index.
      if (isset($paragraph_config['regions'][$index])) {
        $specified_region = $paragraph_config['regions'][$index];

        // Validate that the region exists in the layout.
        if ($this->isValidRegionForLayout($layout_id, $specified_region)) {
          return $specified_region;
        }

        \Drupal::logger('mercury_editor_page_templates')->warning(
          'Invalid region @region for layout @layout_id, using default',
          [
            '@region' => $specified_region,
            '@layout_id' => $layout_id,
          ]
        );
      }
    }

    // Check if individual nested paragraphs specify their region.
    if (isset($paragraph_config['paragraphs'][$index]['region'])) {
      $nested_region = $paragraph_config['paragraphs'][$index]['region'];

      // Validate that the region exists in the layout.
      if ($this->isValidRegionForLayout($layout_id, $nested_region)) {
        return $nested_region;
      }

      \Drupal::logger('mercury_editor_page_templates')->warning(
        'Invalid region @region for layout @layout_id, using default',
        [
          '@region' => $nested_region,
          '@layout_id' => $layout_id,
        ]
      );
    }

    // Use the default region.
    return $default_region;
  }

  /**
   * Validates if a region exists in a layout.
   *
   * @param string $layout_id
   *   The layout ID.
   * @param string $region
   *   The region name.
   *
   * @return bool
   *   TRUE if the region exists, FALSE otherwise.
   */
  protected function isValidRegionForLayout($layout_id, $region) {
    $layout_definition = \Drupal::service('plugin.manager.core.layout')
      ->getDefinition($layout_id);

    if ($layout_definition) {
      $regions = $layout_definition->getRegions();
      return array_key_exists($region, $regions);
    }

    return FALSE;
  }

  /**
   * Builds the HTML for a template card.
   *
   * @param array $template
   *   The template data.
   * @param string $template_id
   *   The template ID.
   *
   * @return string
   *   The HTML markup for the template card.
   */
  protected function buildTemplateCard(array $template, $template_id) {
    // Get the current content type to build the correct URL.
    $request = \Drupal::request();
    $content_type = $this->contentType ?: $request->query->get('content_type');
    if (empty($content_type)) {
      $node_type = \Drupal::routeMatch()->getParameter('node_type');
      if ($node_type) {
        $content_type = $node_type->id();
      }
    }

    $template_url = Url::fromRoute('mercury_editor_page_templates.content_type_templates', [
      'node_type' => $content_type,
    ], [
      'query' => ['template' => $template_id],
    ])->toString();

    $build = [
      '#theme' => 'template_card',
      '#template' => $template,
      '#template_id' => $template_id,
      '#template_url' => $template_url,
    ];

    return $this->renderer->render($build);
  }

}
