<?php

namespace Drupal\mercury_editor_page_templates\Form;

use Drupal\Core\Url;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Extension\ThemeExtensionList;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Mercury Editor Page Builder templates.
 */
class TemplateSettingsForm extends ConfigFormBase {

  /**
   * The YAML serializer.
   *
   * @var \Drupal\Component\Serialization\Yaml
   */
  protected $yaml;

  /**
   * The theme handler service.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * Constructs a TemplateSettingsForm object.
   *
   * @param \Drupal\Component\Serialization\Yaml $yaml
   *   The YAML serializer.
   * @param \Drupal\Core\Extension\ThemeHandlerInterface $theme_handler
   *   The theme handler service.
   */
  public function __construct(Yaml $yaml, ThemeHandlerInterface $theme_handler) {
    $this->yaml = $yaml;
    $this->themeHandler = $theme_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('serialization.yaml'),
      $container->get('theme_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['mercury_editor_page_templates.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'mercury_editor_page_templates_template_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('mercury_editor_page_templates.settings');
    $templates = $config->get('templates') ?: [];

    // Theme configuration section.
    $form['theme_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Theme Settings'),
      '#description' => $this->t('Configure which theme to use for the template selector interface.'),
    ];

    $themes = $this->themeHandler->rebuildThemeData();
    // Remove obsolete themes.
    $themes = array_filter($themes, function ($theme) {
      return !$theme->isObsolete();
    });
    uasort($themes, [ThemeExtensionList::class, 'sortByName']);
    $theme_options = [];

    foreach ($themes as &$theme) {
      if (!empty($theme->status)) {
        $theme_options[$theme->getName()] = $theme->info['name'] . ($theme->isExperimental() ? ' (' . $this->t('Experimental') . ')' : '');
      }
    }

    $form['theme_settings']['template_selector_theme'] = [
      '#type' => 'select',
      '#options' => ['' => $this->t('Default theme')] + $theme_options,
      '#title' => $this->t('Template Selector Theme'),
      '#description' => $this->t('Choose "Default theme" to use the same theme as the admin theme.'),
      '#default_value' => $config->get('template_selector_theme'),
    ];

    // Blank page permissions section.
    $form['blank_page_permissions'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Blank Page Permissions'),
      '#description' => $this->t("Control who can create blank pages using Drupal's permission system."),
    ];

    $form['blank_page_permissions']['permission_info'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('To control who can create blank pages, go to <a href="@permissions_url">Mercury Editor Page Templates Permissions</a> and configure the "Create blank pages without templates" permission.', [
        '@permissions_url' => Url::fromRoute('user.admin_permissions.module', ['modules' => 'mercury_editor_page_templates'])->toString(),
      ]) . '</p>',
    ];

    $form['templates'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Page Templates'),
      '#description' => $this->t('Configure the available page templates for editors. For each template, select which content types it should be available for. No templates are provided by default - you must create them manually.'),
      '#tree' => TRUE,
      '#attributes' => ['id' => 'edit-templates'],
    ];

    foreach ($templates as $id => $template) {
      $form['templates'][$id] = [
        '#type' => 'details',
        '#title' => $template['label'],
        '#open' => !empty($template['is_new']),
      ];

      $form['templates'][$id]['label'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Template Label'),
        '#default_value' => $template['label'],
        '#required' => TRUE,
      ];

      $form['templates'][$id]['description'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Template Description'),
        '#default_value' => $template['description'],
        '#rows' => 2,
      ];

      $form['templates'][$id]['paragraph_reference_field_name'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Paragraph Reference Field Name'),
        '#description' => $this->t('The field name used for page builder content (e.g., field_section, field_content, etc.).'),
        '#default_value' => $template['paragraph_reference_field_name'] ?? '',
        '#required' => TRUE,
      ];

      $form['templates'][$id]['paragraphs'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Template Paragraphs (YAML)'),
        '#default_value' => $this->yaml->encode($template['paragraphs']),
        '#rows' => 10,
        '#description' => $this->t('Define the paragraphs for this template in YAML format.'),
      ];

      // Add content type checkboxes.
      $form['templates'][$id]['content_types'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Available for Content Types'),
        '#description' => $this->t('Select which content types this template should be available for.'),
        '#options' => $this->getContentTypeOptions(),
        '#default_value' => $template['content_types'] ?? [],
        '#required' => TRUE,
      ];

      // Add delete button for each template.
      $form['templates'][$id]['delete'] = [
        '#type' => 'submit',
        '#value' => $this->t('Delete Template'),
        '#submit' => ['::deleteTemplate'],
        '#ajax' => [
          'callback' => '::ajaxDeleteTemplate',
          'wrapper' => 'edit-templates',
          'effect' => 'fade',
        ],
        '#attributes' => [
          'class' => ['button--danger'],
          'onclick' => 'return confirm("' . addslashes($this->t('Are you sure you want to delete this template? This action cannot be undone.')) . '");',
        ],
        '#template_id' => $id,
        '#weight' => 100,
      ];
    }

    // Add "Add New Template" button inside the fieldset.
    $form['templates']['add_new_template'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add New Template'),
      '#submit' => empty($templates) ? ['::addFirstTemplate'] : ['::addNewTemplate'],
      '#ajax' => [
        'callback' => '::ajaxAddTemplate',
        'wrapper' => 'edit-templates',
        'effect' => 'fade',
      ],
      '#weight' => 100,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Check if this is a delete operation.
    $triggering_element = $form_state->getTriggeringElement();
    if (isset($triggering_element['#submit']) && in_array('::deleteTemplate', $triggering_element['#submit'])) {
      // Skip validation for delete operations.
      return;
    }

    // Validate content types for normal form submission.
    $templates_data = $form_state->getValue('templates');
    if (is_array($templates_data)) {
      foreach ($templates_data as $id => $template_data) {
        // Skip if this is not actual form data (e.g., form metadata)
        if (!is_array($template_data) || !isset($template_data['label'])) {
          continue;
        }

        // Safely get the label value - only process if we have form data.
        if (!empty($template_data['label'])) {
          $label = $this->convertToString($template_data['label']);

          $content_types = $template_data['content_types'] ?? [];
          $content_types = array_filter($content_types, function ($value) {
            return $value !== 0;
          });

          if (empty($content_types)) {
            $form_state->setErrorByName("templates][$id][content_types", $this->t('Please select at least one content type for template "@label".', ['@label' => $label]));
          }
          else {
            // Validate that selected content types
            // are configured in Mercury Editor.
            $this->validateMercuryEditorContentTypes($content_types, $form_state, $id, $label);
          }
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('mercury_editor_page_templates.settings');
    $templates = [];

    $templates_data = $form_state->getValue('templates');

    if (is_array($templates_data)) {
      foreach ($templates_data as $id => $template_data) {
        // Skip if this is not actual form data (e.g., form metadata)
        if (!is_array($template_data) || !isset($template_data['label'])) {
          continue;
        }

        // Convert text fields to strings for proper handling.
        $label = $this->convertToString($template_data['label'] ?? '');
        $description = $this->convertToString($template_data['description'] ?? '');
        $paragraph_field = $this->convertToString($template_data['paragraph_reference_field_name'] ?? '');

        if (!empty($label)) {
          try {
            // Handle YAML parsing with better error handling.
            $paragraphs_data = $template_data['paragraphs'] ?? '';
            if (empty($paragraphs_data)) {
              $paragraphs = [];
            }
            else {
              // Try to parse YAML, with fallback for common issues.
              $paragraphs = $this->parseTemplateYaml($paragraphs_data, $id);
            }

            // Process content types checkboxes.
            $content_types = [];
            if (isset($template_data['content_types']) && is_array($template_data['content_types'])) {
              // Filter out unchecked values (0)
              $content_types = array_filter($template_data['content_types'], function ($value) {
                return $value !== 0;
              });
            }

            $templates[$id] = [
              'label' => $label,
              'description' => $description,
              'paragraph_reference_field_name' => $paragraph_field,
              'paragraphs' => $paragraphs,
              'content_types' => $content_types,
            ];

            // Clear the 'is_new' flag when saving.
            unset($templates[$id]['is_new']);
          }
          catch (\Exception $e) {
            \Drupal::logger('mercury_editor_page_templates')->error('YAML decode error for template @id: @error', [
              '@id' => $id,
              '@error' => $e->getMessage(),
            ]);
            $this->messenger()->addError($this->t('Error parsing YAML for template @id: @error', [
              '@id' => $id,
              '@error' => $e->getMessage(),
            ]));
            return;
          }
        }
      }
    }

    \Drupal::logger('mercury_editor_page_templates')->info('Saving templates: @templates', ['@templates' => print_r($templates, TRUE)]);

    // Save theme configuration.
    $theme_value = $form_state->getValue('template_selector_theme');
    $config->set('template_selector_theme', $theme_value);

    $config->set('templates', $templates)->save();

    // Clear any existing status messages to prevent duplicates.
    $this->messenger()->deleteByType('status');
    $this->messenger()->addStatus($this->t('Mercury Editor Page Templates have been saved.'));
  }

  /**
   * Add first template.
   */
  public function addFirstTemplate(array &$form, FormStateInterface $form_state) {
    $config = $this->config('mercury_editor_page_templates.settings');
    $templates = $config->get('templates') ?: [];

    // Add a basic template structure for the user to customize.
    $templates['template_1'] = [
      'label' => 'New Template 1',
      'description' => 'A new template - customize as needed',
      'paragraph_reference_field_name' => 'field_section',
      'paragraphs' => [
        [
          'type' => 'text_block',
          'description' => 'New Section',
          'content' => 'Replace this content with your own.',
        ],
      ],
      // Empty by default - user must select.
      'content_types' => [],
      'is_new' => TRUE,
    ];

    $config->set('templates', $templates)->save();
    $this->messenger()->addStatus($this->t('A basic template has been added. You can now customize it below.'));
    $form_state->setRebuild();
  }

  /**
   * Add new template.
   */
  public function addNewTemplate(array &$form, FormStateInterface $form_state) {
    $config = $this->config('mercury_editor_page_templates.settings');
    $templates = $config->get('templates') ?: [];

    // Generate a unique ID for the new template.
    $template_count = count($templates) + 1;
    $template_id = 'template_' . $template_count;

    // Ensure the ID is unique.
    $counter = 1;
    while (isset($templates[$template_id])) {
      $template_id = 'template_' . $template_count . '_' . $counter;
      $counter++;
    }

    // Add a basic template structure for the user to customize.
    $templates[$template_id] = [
      'label' => 'New Template ' . $template_count,
      'description' => 'A new template - customize as needed',
      'paragraph_reference_field_name' => 'field_section',
      'paragraphs' => [
        [
          'type' => 'text_block',
          'description' => 'New Section',
          'content' => 'Replace this content with your own.',
        ],
      ],
      // Empty by default - user must select.
      'content_types' => [],
      'is_new' => TRUE,
    ];

    $config->set('templates', $templates)->save();
    $this->messenger()->addStatus($this->t('A new template has been added. You can now customize it below.'));
    $form_state->setRebuild();
  }

  /**
   * Delete template.
   */
  public function deleteTemplate(array &$form, FormStateInterface $form_state) {
    $config = $this->config('mercury_editor_page_templates.settings');
    $templates = $config->get('templates') ?: [];

    // Get the template ID from the clicked button.
    $triggering_element = $form_state->getTriggeringElement();
    $template_id = $triggering_element['#template_id'];

    if (isset($templates[$template_id])) {
      $template_label = $templates[$template_id]['label'];
      unset($templates[$template_id]);

      $config->set('templates', $templates)->save();
      $this->messenger()->addStatus($this->t('Template "@label" has been deleted.', ['@label' => $template_label]));
    }
    else {
      $this->messenger()->addError($this->t('Template not found.'));
    }

    $form_state->setRebuild();
  }

  /**
   * AJAX callback for adding templates.
   */
  public function ajaxAddTemplate(array &$form, FormStateInterface $form_state) {
    return $form['templates'];
  }

  /**
   * AJAX callback for deleting templates.
   */
  public function ajaxDeleteTemplate(array &$form, FormStateInterface $form_state) {
    return $form['templates'];
  }

  /**
   * Get available content type options.
   *
   * @return array
   *   Array of content type options keyed by bundle ID.
   */
  protected function getContentTypeOptions() {
    $options = [];
    $node_types = \Drupal::entityTypeManager()->getStorage('node_type')->loadMultiple();

    foreach ($node_types as $node_type) {
      $options[$node_type->id()] = $node_type->label();
    }

    return $options;
  }

  /**
   * Parses template YAML with error recovery and validation.
   *
   * @param string $yaml_data
   *   The YAML data to parse.
   * @param string $template_id
   *   The template ID for error reporting.
   *
   * @return array
   *   The parsed paragraphs array.
   */
  protected function parseTemplateYaml($yaml_data, $template_id) {
    try {
      // First, try to parse the YAML as-is.
      $paragraphs = $this->yaml->decode($yaml_data);

      if (!is_array($paragraphs)) {
        throw new \Exception('YAML does not decode to an array');
      }

      // Validate and fix the structure.
      $paragraphs = $this->validateAndFixParagraphStructure($paragraphs);

      return $paragraphs;
    }
    catch (\Exception $e) {
      \Drupal::logger('mercury_editor_page_templates')->warning(
        'YAML parse error for template @id: @error. Attempting to fix...',
        ['@id' => $template_id, '@error' => $e->getMessage()]
      );

      try {
        $paragraphs = $this->yaml->decode($yaml_data);
        if (is_array($paragraphs)) {
          $paragraphs = $this->validateAndFixParagraphStructure($paragraphs);
          $this->messenger()->addWarning($this->t(
            'Fixed YAML syntax issues in template "@template". Please review and save again.',
            ['@template' => $template_id]
          ));
          return $paragraphs;
        }
      }
      catch (\Exception $e2) {
        \Drupal::logger('mercury_editor_page_templates')->error(
          'Could not fix YAML for template @id: @error',
          ['@id' => $template_id, '@error' => $e2->getMessage()]
        );
      }

      // If all else fails, return empty array.
      $this->messenger()->addError($this->t(
        'Could not parse YAML for template "@template". Please check the syntax and try again.',
        ['@template' => $template_id]
      ));

      return [];
    }
  }

  /**
   * Validates and fixes paragraph structure.
   *
   * @param array $paragraphs
   *   The paragraphs array to validate.
   *
   * @return array
   *   The validated and fixed paragraphs array.
   */
  protected function validateAndFixParagraphStructure($paragraphs) {
    if (!is_array($paragraphs)) {
      return [];
    }

    $fixed_paragraphs = [];

    foreach ($paragraphs as $paragraph) {
      if (!is_array($paragraph)) {
        continue;
      }

      // Ensure required fields exist.
      $fixed_paragraph = [
        'type' => $paragraph['type'] ?? 'text_block',
        'description' => $paragraph['description'] ?? '',
      ];

      // Preserve all other fields dynamically (except the required ones)
      $excluded_fields = ['type', 'description', 'paragraphs'];
      foreach ($paragraph as $field_name => $field_value) {
        if (!in_array($field_name, $excluded_fields)) {
          $fixed_paragraph[$field_name] = $field_value;
        }
      }

      // Handle nested paragraphs.
      if (isset($paragraph['paragraphs']) && is_array($paragraph['paragraphs'])) {
        $fixed_paragraph['paragraphs'] = $this->validateAndFixParagraphStructure($paragraph['paragraphs']);
      }

      $fixed_paragraphs[] = $fixed_paragraph;
    }

    return $fixed_paragraphs;
  }

  /**
   * Validates that content types are configured in Mercury Editor.
   *
   * @param array $content_types
   *   Array of selected content types.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $template_id
   *   The template ID.
   * @param string $template_label
   *   The template label.
   */
  protected function validateMercuryEditorContentTypes(array $content_types, FormStateInterface $form_state, $template_id, $template_label) {
    // Get Mercury Editor configuration.
    $mercury_config = \Drupal::config('mercury_editor.settings');
    $configured_bundles = $mercury_config->get('bundles.node') ?: [];

    $invalid_content_types = [];
    foreach ($content_types as $content_type) {
      if (!isset($configured_bundles[$content_type])) {
        $invalid_content_types[] = $content_type;
      }
    }

    if (!empty($invalid_content_types)) {
      $content_type_list = implode(', ', $invalid_content_types);
      $form_state->setErrorByName("templates][$template_id][content_types",
        $this->t('The following content types are not configured in Mercury Editor: @content_types. Please configure them in <a href="@url">Mercury Editor settings</a> first.', [
          '@content_types' => $content_type_list,
          '@url' => Url::fromRoute('mercury_editor.settings')->toString(),
        ])
      );
    }
  }

  /**
   * Convert a value to string, handling TranslatableMarkup objects.
   *
   * @param mixed $value
   *   The value to convert.
   *
   * @return string
   *   The string representation of the value.
   */
  protected function convertToString($value) {
    // Handle null values.
    if ($value === NULL) {
      return '';
    }

    // Handle TranslatableMarkup objects.
    if (is_object($value)) {
      if (method_exists($value, '__toString')) {
        return (string) $value;
      }
      // If an object without __toString, try to get its string representation.
      if (method_exists($value, 'render')) {
        return (string) $value->render();
      }
      // Last resort: convert to string.
      return (string) $value;
    }

    // Handle arrays (shouldn't happen, but just in case)
    if (is_array($value)) {
      return '';
    }

    // Handle primitive values.
    return (string) $value;
  }

}
